{
//////////////////////////////////////////////////////////////////////
// file FC3Lib.pas
//
// contains all of the functions needed for working with
// the FuzzyCOPE 3 engine
//
// last modified on: 14/05/99
// last modified by: FuzzyCOPE 3 Team (fuzzycope@otago.ac.nz)
//////////////////////////////////////////////////////////////////////
}

unit FC3Lib;

interface
uses Dialogs, SysUtils, ParseFunction;

{DLL interface functions}
function fcSendCommand ( StgMessage : AnsiString ) : integer;
function fcGetErrorString : AnsiString;
function fcGetResultString : AnsiString;

{object management functions}
function CopyObject ( StgSource : AnsiString; StgDestination : AnsiString ) : integer;
function DeleteObject ( StgAlias : AnsiString ) : integer;
function PurgeAllObjects : integer;
function AliasExists ( StgAlias : AnsiString ) : boolean;
function ObjectType ( StgAlias : AnsiString; var StgType : AnsiString ) : integer;

{MLP Functions}
function CreateMLP ( StgAlias : AnsiString; IntLayers : integer; DyaIntLayerSize : Variant; BlnBias : boolean ) : integer;
function GetMLPStructure ( StgAlias : AnsiString; var IntLayers : integer; var IntInputs : integer; var DyaIntHidden : Variant; var IntOutputs : integer; var BlnBias : boolean ) : integer;
function SaveMLP ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;
function LoadMLP ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;
function BPTrainMLP ( StgAlias : AnsiString; StgDataAlias : AnsiString; IntEpochs : integer; FltLearningRate : real; FltMomentum : real; FltTerminating : real; BlnBatch : boolean ) : integer;

{FuNN Functions}
function CreateFuNN ( StgAlias : AnsiString; IntInputs : integer; DyaIntCondition : Variant; IntRules : integer; DyaIntAction : Variant; IntOutputs : integer ) : integer;
function GetFuNNStructure ( StgAlias : AnsiString; var IntInputs : integer; var DyaIntCondition : Variant; var IntRules : integer; var DyaIntAction : Variant; var IntOutputs : integer ) : integer;
function SaveFuNN ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;
function LoadFuNN ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;
function CreateLWFTrainer ( StgAlias : AnsiString; StgNetwork : AnsiString ) : integer;
function SaveFuNNLabels ( StgAlias : AnsiString; StgFilename : AnsiString ) : integer;
function LoadFuNNLabels ( StgAlias : AnsiString; StgFilename : AnsiString ) : integer;
function LabelFuNN ( StgNetwork : AnsiString; StgLabels : AnsiString ) : integer;
function REFuNN ( StgNetwork : AnsiString; StgLabels : AnsiString; StgRules : AnsiString; FltThreshIn : real; FltThreshOut : real ) : integer;
function RIFuNN ( StgRules : AnsiString; StgNetwork : AnsiString ) : integer;
function SaveFuzzyRules ( StgAlias : AnsiString; StgFilename : AnsiString ) : integer;
function LoadFuzzyRules ( StgAlias : AnsiString; StgFilename : AnsiString ) : integer;

{Kohonen Functions}
function CreateKohonen ( StgAlias : AnsiString; IntInputs : integer; IntDimensions : integer; DyaIntMapDim : Variant ) : integer;
function GetKohonenStructure ( StgAlias : AnsiString; var IntInputs : integer; var IntDimensions : integer; var DyaIntMapDim : Variant ) : integer;
function CreateKohonenTrainer ( StgAlias : AnsiString; StgNetwork : AnsiString ) : integer;
function TrainKohonen ( StgAlias : AnsiString; StgData : AnsiString; IntEpochs : integer; FltLearningRate : real; DyaIntNeighbourhood : Variant ) : integer;
function MapKohonen ( StgNetwork : AnsiString; StgInput : AnsiString; StgOutput : AnsiString ) : integer;
function LabelKohonen ( StgNetwork : AnsiString; StgMap : AnsiString; StgCoord : AnsiString; StgOutput : AnsiString ) : integer;
function SaveKohonen ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;
function LoadKohonen ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;

{General ANN Functions}
function CreateBPTrainer ( StgAlias : AnsiString; StgNetwork : AnsiString ) : integer;
function CreateGATrainer ( StgAlias : AnsiString; StgNetwork : AnsiString; FltMinweight : real; FltMaxWeight : real; IntPopSize : integer ) : integer;
function BPTrainANN ( StgAlias : AnsiString; StgDataAlias : AnsiString; IntEpochs : integer; FltLearningRate : real; FltMomentum : real; FltTerminating : real; BlnBatch : boolean ) : integer;
function GATrainANN ( StgAlias : AnsiString; StgData : AnsiString; IntGenerations : integer; FltMutation : real; IntSelection : integer; IntCrossover : integer; BlnElitism : boolean; BlnNormalisation : boolean; FltTerminating : real ) : integer;
function LWFTrainANN( StgAlias : AnsiString; StgDataAlias : AnsiString; IntEpochs : integer; FltLearningRate : real; FltMomentum : real; FltDecay : real; FltTerminating : real; BlnBatch : boolean ) : integer;
function Recall ( StgAlias : AnsiString; StgInput : AnsiString; StgOutput : AnsiString ) : integer;
function GetTrained ( StgSource : AnsiString; StgDestination : AnsiString ) : integer;

{Data object functions}
function CreateData ( StgAlias : AnsiString; IntInputs : integer; IntOutputs : integer; IntRows : integer; BlnLabels : boolean ) : integer;
function GetDataStructure ( StgAlias : AnsiString; var IntInputs : integer; var IntOutputs : integer; var IntRows : integer; var IntDataRows : integer; var BlnLabels : boolean ) : integer;
function AppendRow ( StgAlias : AnsiString; DyaFltInputs : Variant; DyaFltOutputs : Variant; DyaStgLabels : Variant; DyaFltLabelWeights : Variant; BlnInputs : boolean; BlnOutputs : boolean; BlnLabels : boolean ) : integer;
function PurgeData ( StgAlias : AnsiString ) : integer;
function ResetData ( StgAlias : AnsiString ) : integer;
function GetNextRow ( StgAlias : AnsiString; var DyaFltInputs : Variant; var DyaFltOutputs : Variant; var DyaStgLabels : Variant; var DyaFltLabelWeights : Variant; var IntInputs : integer; var IntOutputs : integer; var IntNumLabels : integer ) : integer;
function SaveData ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;
function LoadData ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;

{data transformation functions}
function ShuffleData ( StgSource : AnsiString; StgDestination : AnsiString ) : integer;
function LinearNormaliseInputs ( StgSource : AnsiString; StgDestination : AnsiString; DyaFltMins : Variant; DyaFltMaxs : Variant ) : integer;
function LinearNormaliseOutputs ( StgSource : AnsiString; StgDestination : AnsiString; DyaFltMins : Variant; DyaFltMaxs : Variant ) : integer;
function LinearNormaliseAll ( StgSource : AnsiString; StgDestination : AnsiString; DyaFltInputMins : Variant; DyaFltInputMaxs : Variant; DyaFltOutputMins : Variant; DyaFltOutputMaxs : Variant ) : integer;

{data statistics functions}
function CreateDataStats ( StgAlias : AnsiString; StgData : AnsiString ) : integer;
function GetInputMaximums ( StgAlias : AnsiString; var DyaFltInputMax : Variant; var IntColumns : integer ) : integer;
function GetOutputMaximums ( StgAlias : AnsiString; var DyaFltOutputMax : Variant; var IntColumns : integer ) : integer;
function GetMaximums ( StgAlias : AnsiString; var DyaFltInputMax : Variant; var IntInputColumns : integer; var DyaFltOutputMax : Variant; var IntOutputColumns : integer ) : integer;
function GetInputMinimums ( StgAlias : AnsiString; var DyaFltInputMin : Variant; var IntColumns : integer ) : integer;
function GetOutputMinimums ( StgAlias : AnsiString; var DyaFltOutputMin : Variant; var IntColumns : integer ) : integer;
function GetMinimums ( StgAlias : AnsiString; var DyaFltInputMin : Variant; var IntInputColumns : integer; var DyaFltOutputMin : Variant; var IntOutputColumns : integer ) : integer;
function GetInputAverages ( StgAlias : AnsiString; var DyaFltInputAverages : Variant; var IntColumns : integer ) : integer;
function GetOutputAverages ( StgAlias : AnsiString; var DyaFltOutputAverages : Variant; var IntColumns : integer ) : integer;
function GetAverages ( StgAlias : AnsiString; var DyaFltInputAverages : Variant; var IntInputColumns : integer; var DyaFltOutputAverages : Variant; var IntOutputColumns : integer ) : integer;

{miscellaneous functions}
function StartLogging ( StgFilename : AnsiString ) : integer;
function StopLogging : integer;
function RunScript ( StgFilename : AnsiString ) : integer;

const

  TOURNAMENT = 1;
  ROULETTE = 2;
  
implementation



{DLL function declarations}
{these get called from the fcSendCommand, fcGetErrorString and fcGetResultString functions}

function _fcSendCommand ( const TheValue : PChar ) : integer; far; stdcall; external 'FC3.dll';
function _fcGetErrorString : PChar; far; stdcall; external 'FC3.dll';
function _fcGetResultString : PChar; far; stdcall; external 'FC3.dll';


{
//////////////////////////////////////////////////////////////////////
// function fcSendCommand
//
// sends command strings to the engine
//
//////////////////////////////////////////////////////////////////////
}

function fcSendCommand ( StgMessage : AnsiString ) : integer;

var

StgPassString : PChar;
IntLength : integer;
IntReturn : integer;

begin

  IntLength := length ( StgMessage ) + 1;
  GetMem ( StgPassString, IntLength );
  StrPCopyWorks ( StgPassString, StgMessage ); 
  IntReturn := _fcSendCommand ( StgPassString );
  FreeMem ( StgPassString );

  fcSendCommand := IntReturn;

end;


{
//////////////////////////////////////////////////////////////////////
// function fcGetErrorString
//
// returns the last error message from the engine
//
//////////////////////////////////////////////////////////////////////
}

function fcGetErrorString : AnsiString;

begin

  fcGetErrorString := _fcGetErrorString;

end;


{
//////////////////////////////////////////////////////////////////////
// function fcGetResultString
//
// returns the last result string from the engine
//
//////////////////////////////////////////////////////////////////////
}

function fcGetResultString : AnsiString;

begin

  fcGetResultString := _fcGetResultString;

end; {fcGetResultString}


{
//////////////////////////////////////////////////////////////////////
// object management functions
//////////////////////////////////////////////////////////////////////
}

{
//////////////////////////////////////////////////////////////////////
// function CopyObject
//
// copies the specified object
//
// StgSource = alias of original object
// StgDestination = alias to assign copy to
//
//////////////////////////////////////////////////////////////////////
}

function CopyObject ( StgSource : AnsiString; StgDestination : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'copy source ' + StgSource + ' destination ' + StgDestination;
  CopyObject := fcSendCommand ( StgCommand );

end; {CopyObject}


{
//////////////////////////////////////////////////////////////////////
// function DeleteObject
//
// removes the associated object and fres the alias
//
// StgAlias = Alias to delete
//
//////////////////////////////////////////////////////////////////////
}

function DeleteObject ( StgAlias : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'destroy ' + StgAlias;
  DeleteObject := fcSendCommand ( StgCommand  );

end; {DeleteObject}


{
//////////////////////////////////////////////////////////////////////
// function PurgeAllObjects
//
// removes all objects from memory and frees all
// aliases 
//
//////////////////////////////////////////////////////////////////////
}

function PurgeAllObjects : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'purge';
  PurgeAllObjects := fcSendCommand ( StgCommand );

end; {PurgeAllObjects}


{
//////////////////////////////////////////////////////////////////////
// function AliasExists
//
// returns 'true' if the alias exists, otherwise
// false
//
// StgAlias = alias to query
//
//////////////////////////////////////////////////////////////////////
}

function AliasExists ( StgAlias : AnsiString ) : boolean;

var

  StgQuery : AnsiString;

begin

  StgQuery := 'query aliasexists ' + StgAlias;
  fcSendCommand ( StgQuery );
  
  if fcGetResultString = 'true' then
    AliasExists := true
  else
    AliasExists := false;

end; {AliasExists}


{
//////////////////////////////////////////////////////////////////////
// function ObjectType 
//
// returns the type of the specified object
//
// StgAlias : alias of object 
// StgType : AnsiString type is returned into
//
//////////////////////////////////////////////////////////////////////
}

function ObjectType ( StgAlias : AnsiString; var StgType : AnsiString ) : integer;

var

  StgQuery : AnsiString;
  IntReturn : integer;

begin

  StgQuery := 'query typeof ' + StgAlias;
  IntReturn := fcSendCommand ( StgQuery );
  if IntReturn = SUCCESS then
    StgType := fcGetResultString
  else
    StgType := '';

  ObjectType := IntReturn;

end; {ObjectType}


{
//////////////////////////////////////////////////////////////////////
//MLP Functions
//////////////////////////////////////////////////////////////////////
}

{
//////////////////////////////////////////////////////////////////////
// function CreateMLP
//
// creates and initialises an MLP ANN
//
// StgAlias = Alias to assign object to
// IntLayers = layers in the MLP
// DyaIntLayerSize = array containing the size of 
//   each layer
//
//////////////////////////////////////////////////////////////////////
}

function CreateMLP ( StgAlias : AnsiString; IntLayers : integer; DyaIntLayerSize : Variant; BlnBias : boolean ) : integer;

var

  StgCommand : AnsiString;
  StgLayers : AnsiString;
  StgInputs : AnsiString;
  StgOutputs : AnsiString;
  IntHighBound : integer;
  IntLowBound : integer;
  IntCount : integer;
  IntLastItem : integer;
  IntCountStart : integer;
  IntCountEnd : integer;

begin

  if VarArrayDimCount ( DyaIntLayerSize ) <> 1 then
    raise Exception.Create('One-dimensional variant array expected');

  IntLowBound := VarArrayLowBound ( DyaIntLayerSize, 1 );
  IntHighBound := VarArrayHighBound ( DyaIntLayerSize, 1 );

  {assemble the command string}
  StgLayers := IntToStr ( IntLayers );
  StgInputs := IntToStr ( DyaIntLayerSize [ IntLowBound ] );
  StgOutputs := IntToStr ( DyaIntLAyerSize [ IntHighBound ] );
  StgCommand := 'create MLP alias ' + StgAlias + ' layers ' + StgLayers + ' inputs ' + StgInputs + ' hidden ' + LIST_START;

  {assemble the hidden layers array}
  IntLastItem := IntHighBound - 1;
  IntCountStart := IntLowBound + 1;
  IntCountEnd := IntHighBound - 1;
  for IntCount := IntCountStart to IntCountEnd do    
  begin
    StgCommand := StgCommand + IntToStr ( DyaIntLayerSize [ IntCount ] );
    if IntCount <> IntLastItem then
    begin
      StgCommand := StgCommand + ITEM_DELIM;
      StgCommand := StgCommand + ' ';
    end; {if}
  end; {for}  
 
  StgCommand := StgCommand + LIST_END + ' outputs ' + StgOutputs + ' bias ';
  if BlnBias then
    StgCommand := StgCommand + 'on'
  else
    StgCommand := StgCommand + 'off';

  {assembled the command string, so send it}
  CreateMLP := fcSendCommand ( StgCommand ); 
    
end; {CreateMLP}


{
//////////////////////////////////////////////////////////////////////
// function GetMLPStructure
//
// returns thes tructure of an MLP object
//
// StgAlias : alias of MLP
// IntLayers : number of layers in the MLP
// IntInputs : number of input nodes
// DyaIntHidden : one dimensional variant array of int that the size
//                of each hidden layer is returned into
// IntOutputs : number of output nodes
// BlnBias : whether or not the MLP has a bias layer
//
//////////////////////////////////////////////////////////////////////
}

function GetMLPStructure ( StgAlias : AnsiString; var IntLayers : integer; var IntInputs : integer; var DyaIntHidden : Variant; var IntOutputs : integer; var BlnBias : boolean ) : integer;

var

  StgQuery : AnsiString;
  IntReturn : integer;
  StgResult : AnsiString;
  StgPResult : PChar;
  IntResultLength : integer;
  IntHiddenLayers : integer;

begin

  {check it is an MLP}
  StgQuery := 'query classof ' + StgAlias;
  IntReturn := fcSendCommand ( StgQuery );
  if IntReturn >= 0 then
  begin
    StgResult := fcGetResultString;
    if StgResult = 'MLP' then 
    begin
      {assemble the query to get the structure}
      StgQuery := 'query structureof ' + StgAlias;
      IntReturn := fcSendCommand ( StgQuery );
      if IntReturn >= 0 then      
      begin
        {retrieve and parse the result string}
        StgResult := fcGetResultString;
        IntResultLength := length ( StgResult ) + 1;
        GetMem ( StgPResult, IntResultLength );
        StrPCopyWorks ( StgPResult, StgResult );
        IntReturn := GetArgumentInt ( StgPResult, 'layers', IntLayers );
        if IntReturn <> SUCCESS then
          raise Exception.Create('Internal parsing error');
        IntReturn := GetArgumentInt ( StgPResult, 'inputs', IntInputs );
        if IntReturn <> SUCCESS then
          raise Exception.Create('Internal parsing error');
    
        IntReturn := GetIntArguments ( StgPResult, 'hidden', DyaIntHidden, IntHiddenLayers );
        if IntReturn <> SUCCESS then
          raise Exception.Create('Internal parsing error');
    
        IntReturn := GetArgumentInt ( StgPResult, 'outputs', IntOutputs );
        if IntReturn <> SUCCESS then
          raise Exception.Create('Internal parsing error');
    
        IntReturn := GetArgumentBln ( StgPResult, 'bias', BlnBias );
        if IntReturn <> SUCCESS then
          raise Exception.Create('Internal parsing error');
        FreeMem ( StgPResult );
      end {if}
      else
        IntReturn := IntReturn;
    end {if}
    else
      IntReturn := INCORRECT_OBJECT_TYPE;
  end; {if}

  GetMLPStructure := IntReturn;

end; {GetMLPStructure}


{
//////////////////////////////////////////////////////////////////////
// function SaveMLP
//
// Saves an MLP ANN to disk file
//
// StgAlias = Alias of MLP to save
// StgFileName = name of file to save to
//
//////////////////////////////////////////////////////////////////////
}

function SaveMLP ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'save MLP alias ' + StgAlias + ' filename ' + START_QUOTE + StgFileName + END_QUOTE;
  SaveMLP := fcSendCommand ( StgCommand );

end; {SaveMLP}


{
//////////////////////////////////////////////////////////////////////
// function LoadMLP
//
// Loads an MLP ANN from disk file
//
// StgAlias = Alias to assign MLP to 
// StgFileName = name of file to load
//
//////////////////////////////////////////////////////////////////////
}

function LoadMLP ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'load MLP alias ' + StgAlias + ' filename ' + START_QUOTE + StgFileName + END_QUOTE;
  LoadMLP := fcSendCommand ( StgCommand );

end; {LoadMLP}


function BPTrainMLP ( StgAlias : AnsiString; StgDataAlias : AnsiString; IntEpochs : integer; FltLearningRate : real; FltMomentum : real; FltTerminating : real; BlnBatch : boolean ) : integer;

begin

  BPTrainMLP := BPTrainANN ( StgAlias, StgDataAlias, IntEpochs, FltLearningRate, FltMomentum, FltTerminating, BlnBatch );

end;


{
//////////////////////////////////////////////////////////////////////
// FuNN Functions
//////////////////////////////////////////////////////////////////////
}


{
//////////////////////////////////////////////////////////////////////
// function CreateFuNN
//
// creates and initialises a FuNN
//
// StgAlias = alias to assign FuNN object to
// IntInputs = number of input nodes
// DyaIntCondition = condition nodes per input
// IntRules = number of rule nodes
// DyaIntAction = action nodes per output
// IntOutputs = number of output nodes
//
//////////////////////////////////////////////////////////////////////
}

function CreateFuNN ( StgAlias : AnsiString; IntInputs : integer; DyaIntCondition : Variant; IntRules : integer; DyaIntAction : Variant; IntOutputs : integer ) : integer;

var

  StgCommand : AnsiString;
  StgInputs : AnsiString;
  StgRules : AnsiString;
  StgOutputs : AnsiString;
  IntCondLowBound : integer;
  IntCondHighBound : integer;
  IntActLowBound : integer;
  IntActHighBound : integer;
  IntCount : integer;
  
begin

  {throw an exception if the arrays are the wrong dimension}
  if VarArrayDimCount ( DyaIntCondition ) <> 1 then
    raise Exception.Create('One-dimensional variant array expected');
  if VarArrayDimCount ( DyaIntAction ) <> 1 then
    raise Exception.Create('One-dimensional variant array expected');

  {retrieve the bounds of the two arrays}
  IntCondLowBound := VarArrayLowBound ( DyaIntCondition, 1 );
  IntCondHighBound := VarArrayHighBound ( DyaIntCondition, 1 );
  IntActLowBound := VarArrayLowBound ( DyaIntAction, 1 );
  IntActHighBound := VarArrayHighBound ( DyaIntAction, 1 );

  {assemble the command string}
  StgInputs := IntToStr ( IntInputs );
  StgRules := IntToStr ( IntRules );
  StgOutputs := IntToStr ( IntOutputs );
  StgCommand := 'create FuNN alias ' + StgAlias + ' inputs ' + StgInputs + ' condition ' + LIST_START;
  
  {insert the condition node information}
  for IntCount := IntCondLowBound to IntCondHighBound - 1 do
  begin
    StgCommand := StgCommand + IntToStr ( DyaIntCondition [ IntCount ] );
    StgCommand := StgCommand + ITEM_DELIM;
    StgCommand := StgCommand + ' ';
  end; {if}
  StgCommand := StgCommand + IntToStr ( DyaIntCondition [ IntCondHighBound ] );

  StgCommand := StgCommand + LIST_END + ' rules ' + StgRules + ' action ' + LIST_START;

  {insert the action node information}
  for IntCount := IntActLowBound to IntActHighBound - 1 do
  begin
    StgCommand := StgCommand + IntToStr ( DyaIntAction [ IntCount ] );
    StgCommand := StgCommand + ITEM_DELIM;
    StgCommand := StgCommand + ' ';
  end; {if}
  StgCommand := StgCommand + IntToStr ( DyaIntAction [ IntActHighBound ] );
  StgCommand := StgCommand + LIST_END + ' outputs ' + StgOutputs;

 {completed assembling the command string, so send it}
 CreateFuNN := fcSendCommand ( StgCommand );

end; {CreateFuNN} 


{
//////////////////////////////////////////////////////////////////////
// function GetFuNNStructure
// 
// returns the structure of the specified FuNN
//
// StgAlias : alias of the FuNN
// IntInputs : number of input nodes
// DyaIntCondition : one dimensional variant array of int the specifies
//                   how many condition nodes are attached to each input
// IntRules : number of rule nodes
// DyaIntAction : one dimensional variant array of int the specifies
//                   how many action nodes are attached to each output
// IntOutputs : number of outputs nodes
//
//////////////////////////////////////////////////////////////////////
}

function GetFuNNStructure ( StgAlias : AnsiString; var IntInputs : integer; var DyaIntCondition : Variant; var IntRules : integer; var DyaIntAction : Variant; var IntOutputs : integer ) : integer;

var

  StgQuery : AnsiString;
  StgResult : AnsiString;
  StgPResult : PChar;
  IntResultLength : integer;
  IntReturn : integer;
  IntCount : integer;

begin

  {check it is a FuNN}
  StgQuery := 'query classof ' + StgAlias;
  IntReturn := fcSendCommand ( StgQuery );
  if IntReturn >= 0 then
  begin
    StgResult := fcGetResultString;
    if StgResult = 'FuNN' then
    begin
      {assemble the query to get the structure}
      StgQuery := 'query structureof ' + StgAlias;
      IntReturn := fcSendCommand ( StgQuery );
      if IntReturn >= 0 then
      begin
        {retrive and parse the result string}
        StgResult := fcGetResultString;
        IntResultLength := length ( StgResult ) + 1;
        GetMem ( StgPResult, IntResultLength );
        StrPCopyWorks ( StgPResult, StgResult );
        IntReturn := GetArgumentInt ( StgPResult, 'inputs', IntInputs );
        if IntReturn <> SUCCESS then
          raise Exception.Create('Internal parsing error');
        IntReturn := GetIntArguments ( StgPResult, 'condition', DyaIntCondition, IntCount );
        if IntReturn <> SUCCESS then
          raise Exception.Create('Internal parsing error');
        IntReturn := GetArgumentInt ( StgPResult, 'rules', IntRules );
        if IntReturn <> SUCCESS then
          raise Exception.Create('Internal parsing error');
        IntReturn := GetIntArguments ( StgPResult, 'action', DyaIntAction, IntCount );
        if IntReturn <> SUCCESS then
          raise Exception.Create('Internal parsing error');
        IntReturn := GetArgumentInt ( StgPResult, 'outputs', IntOutputs );
        if IntReturn <> SUCCESS then
          raise Exception.Create('Internal parsing error');
        FreeMem ( StgPResult );
      end {if}
    end {if}
    else
      IntReturn := INCORRECT_OBJECT_TYPE;
  end; {if}

  GetFuNNStructure := IntReturn;

end; {GetFuNNStructure}


{
//////////////////////////////////////////////////////////////////////
// function SaveFuNN
//
// saves the FuNN associated with the alias to 
// the specified file
//
// StgAlias = alias of the FuNN
// StgFileName = file to save to
//
//////////////////////////////////////////////////////////////////////
}

function SaveFuNN ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'save FuNN alias ' + StgAlias + ' filename ' + START_QUOTE + StgFileName + END_QUOTE;
  SaveFuNN := fcSendCommand ( StgCommand );

end; {SaveFuNN}


{
//////////////////////////////////////////////////////////////////////
// function LoadFuNN
//
// loads a FuNN from disk
//
// StgAlias = the alias to assign the FuNN to
// StgFilename = name of file to load
//
//////////////////////////////////////////////////////////////////////
}

function LoadFuNN ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'load FuNN alias ' + StgAlias + ' filename ' + START_QUOTE + StgFileName + END_QUOTE;
  LoadFuNN := fcSendCommand ( StgCommand );

end; {LoadFuNN}


{
//////////////////////////////////////////////////////////////////////
// function CreateLWFTrainer
//
// creates a Learning With Forgetting trainer object for a FuNN
//
// StgAlias : alias to assign trainer to
// StgNetwork : alias of FuNN network
//
//////////////////////////////////////////////////////////////////////
}

function CreateLWFTrainer ( StgAlias : AnsiString; StgNetwork : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'create LWFTrainer alias ' + StgAlias + ' network ' + StgNetwork;
  CreateLWFTrainer := fcSendCommand ( StgCommand );

end; {CreateLWFTrainer}


{
//////////////////////////////////////////////////////////////////////
// function LabelFuNN
//
// generates a FuNNLabel object for the specified FuNN network
//
// StgNetwork : alias of the FuNN to label
// StgLabels : alias to assign labels to
//
//////////////////////////////////////////////////////////////////////
}

function LabelFuNN ( StgNetwork : AnsiString; StgLabels : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'labelfunn network ' + StgNetwork;
  StgCommand := StgCommand + ' labels ' + StgLabels;
  LabelFuNN := fcSendCommand ( StgCommand );

end; {LabelFuNN}


{
//////////////////////////////////////////////////////////////////////
// function SaveFuNNLabels
//
// saves a FuNNLabel object to file
//
// StgAlias : alias of the FuNNLabel object to save
// StgFilename : name of file to save to
//
//////////////////////////////////////////////////////////////////////
}

function SaveFuNNLabels ( StgAlias : AnsiString; StgFilename : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'save FuNNLabels alias ' + StgAlias + ' filename ';
  StgCommand := StgCommand + START_QUOTE + StgFilename + END_QUOTE;
  SaveFuNNLabels := fcSendCommand ( StgCommand );

end; {SaveFuNNLabels}


{
//////////////////////////////////////////////////////////////////////
// function LoadFuNNLabels
//
// loads a FuNNLabel object from file
//
// StgAlias : alias to assign the FuNNLabel object to
// StgFilename : name of file to load
//
//////////////////////////////////////////////////////////////////////
}

function LoadFuNNLabels ( StgAlias : AnsiString; StgFilename : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'load FuNNLabels alias ' + StgAlias + ' filename ';
  StgCommand := StgCommand + START_QUOTE + StgFilename + END_QUOTE;
  LoadFuNNLabels := fcSendCommand ( StgCommand );

end; {LoadFuNNLabels}


{
//////////////////////////////////////////////////////////////////////
// function REFuNN
//
// extracts fuzzy rules from a FuNN
//
// StgNetwork : alias of network to extract rules from
// StgLabels : alias of FuNNLabel object to use to label rules
// StgRules : alias to assign FuzzyRules object to
// FltThreshIn : incoming weights threshold
// FltThreshOut : outgoing weight threshold
//
//////////////////////////////////////////////////////////////////////
}

function REFuNN ( StgNetwork : AnsiString; StgLabels : AnsiString; StgRules : AnsiString; FltThreshIn : real; FltThreshOut : real ) : integer;

var

  StgCommand : AnsiString;
  StgThreshIn : AnsiString;
  StgThreshOut : AnsiString;

begin

  StgThreshIn := FloatToStr ( FltThreshIn );
  StgThreshOut := FloatToStr ( FltThreshOut );
  StgCommand := 'refunn network ' + StgNetwork + ' labels ' + StgLabels;
  StgCommand := StgCommand + ' rules ' + StgRules + ' ThresholdIn ' + StgThreshIn;
  StgCommand := StgCommand + ' ThresholdOut ' + StgThreshOut;
  REFuNN := fcSendCommand ( StgCommand );

end; {REFuNN}


{
//////////////////////////////////////////////////////////////////////
// function RIFuNN
//
// create a FuNN from a set of fuzzy rules
//
// StgRules : alias of rules object
// StgNetwork : alisa to assign FuNN to
//
//////////////////////////////////////////////////////////////////////
}

function RIFuNN ( StgRules : AnsiString; StgNetwork : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'rifunn rules ' + StgRules + ' network ' + StgNetwork;
  RIFuNN := fcSendCommand ( StgCommand );

end; {RIFuNN}


{
//////////////////////////////////////////////////////////////////////
// function SaveFuzzyRules
//
// saves a FuzzyRules object to file
//
// StgAlias : alias of rules object to save
// StgFilename : name of file to save to
//
//////////////////////////////////////////////////////////////////////
}

function SaveFuzzyRules ( StgAlias : AnsiString; StgFilename : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'save FuzzyRules alias ' + StgAlias + ' filename ';
  StgCommand := StgCommand + START_QUOTE + StgFilename + END_QUOTE;
  SaveFuzzyRules := fcSendCommand ( StgCommand );

end; {SaveFuzzyRules}


{
//////////////////////////////////////////////////////////////////////
// function LoadFuzzyRules
//
// loads a FuzzyRules object from file
//
// StgAlias : alias to assign rules to
// StgFilename : name of file to load from
//
//////////////////////////////////////////////////////////////////////
}

function LoadFuzzyRules ( StgAlias : AnsiString; StgFilename : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'load FuzzyRules alias ' + StgAlias + ' filename ';
  StgCommand := StgCommand + START_QUOTE + StgFilename + END_QUOTE;
  LoadFuzzyRules := fcSendCommand ( StgCommand );

end; {LoadFuzzyRules}


{
//////////////////////////////////////////////////////////////////////
// Kohonen Functions
//////////////////////////////////////////////////////////////////////
}


{
//////////////////////////////////////////////////////////////////////
// function CreateKohonen
//
// creates a Kohonen network
//
// StgAlias : alias to assign Kohonen to
// IntInputs : number of input nodes
// IntDimensions : Number of dimensions in output map
// DyaIntMapDim : one dimensional variant array of integer containing 
//                the size of each dimensions in the map
//
//////////////////////////////////////////////////////////////////////
}

function CreateKohonen ( StgAlias : AnsiString; IntInputs : integer; IntDimensions : integer; DyaIntMapDim : Variant ) : integer;

var

  StgCommand : AnsiString;
  StgInputs : AnsiString;
  StgDimensions : AnsiString;
  IntCount : integer;
  IntHighBound : integer;
  IntLowBound : integer;

begin

  {check the parameters}
  {throw an exception if the arrays are the wrong dimension}
  if VarArrayDimCount ( DyaIntMapDim ) <> 1 then
    raise Exception.Create('One-dimensional variant array expected');

  {retrieve the bounds of the array}
  IntLowBound := VarArrayLowBound ( DyaIntMapDim, 1 );
  IntHighBound := VarArrayHighBound ( DyaIntMapDim, 1 );

  StgInputs := IntToStr ( IntInputs );
  StgDimensions := IntToStr ( IntDimensions );
  StgCommand := 'create Kohonen alias ' + StgAlias + ' inputs ' + StgInputs;
  StgCommand := StgCommand + ' dimensions ' + StgDimensions + ' mapdim ' + LIST_START;
  for IntCount := IntLowBound to IntHighBound - 1 do
  begin
    StgCommand := StgCommand + IntToStr ( DyaIntMapDim [ IntCount ] ); 
    StgCommand := StgCommand + ITEM_DELIM;
    StgCommand := StgCommand + ' ';
  end; {for}
  StgCommand := StgCommand + IntToStr ( DyaIntMapDim [ IntHighBound ] );
  StgCommand := StgCommand + LIST_END;

  {completed assembling the string, so send it}
  CreateKohonen := fcSendCommand ( StgCommand );

end; {CreateKohonen}


{
//////////////////////////////////////////////////////////////////////
// GetKohonenStructure
//
// returns the structure of an existing Kohonen net
//
// StgAlias : alias of Kohonen network
// IntInputs : number of input nodes
// IntDimensions : Number of dimensions in output map
// DyaIntMapDim : one dimensional variant array of integer containing 
//                the size of each dimensions in the map
//
//////////////////////////////////////////////////////////////////////
}

function GetKohonenStructure ( StgAlias : AnsiString; var IntInputs : integer; var IntDimensions : integer; var DyaIntMapDim : Variant ) : integer;

var

  StgCommand : AnsiString;
  StgResult : AnsiString;
  StgPResult : PChar;
  IntResultLength : integer;
  IntDimensionItems : integer;
  IntReturn : integer;

begin

  StgCommand := 'query structureof ' + StgAlias;
  IntReturn := fcSendCommand ( StgCommand );
  if IntReturn >= 0 then
  begin
    {retrieve and parse the results string}
    StgResult := fcGetResultString;
    IntResultLength := length ( StgResult ) + 1;
    GetMem ( StgPResult, IntResultLength );
    StrPCopyWorks ( StgPResult, StgResult );
    IntReturn := GetArgumentInt ( StgPResult, 'inputs', IntInputs );
    if IntReturn <> SUCCESS then
      raise Exception.Create('Internal parsing error');
    IntReturn := GetArgumentInt ( StgPResult, 'dimensions', IntDimensions );
    if IntReturn <> SUCCESS then
      raise Exception.Create('Internal parsing error');
    IntReturn := GetIntArguments ( StgPResult, 'mapdim', DyaIntMapDim, IntDimensionItems );
    if IntReturn <> SUCCESS then
      raise Exception.Create('Internal parsing error');
    if IntDimensions <> IntDimensionItems then
      raise Exception.Create('Internal parsing error');
    FreeMem ( StgPResult );
  end; {if}
  
  GetKohonenStructure := IntReturn;

end; {GetKohonenStructure}


{
//////////////////////////////////////////////////////////////////////
// function CreateKohonenTrainer
//
// creates a Kohonen training object
//
// StgAlias : alias to assign trainer to
// StgNetwork 
//
//////////////////////////////////////////////////////////////////////
}

function CreateKohonenTrainer ( StgAlias : AnsiString; StgNetwork : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'create KohonenTrainer alias ' + StgAlias + ' network ' + StgNetwork;
  CreateKohonenTrainer := fcSendCommand ( StgCommand );

end; {CreateKohonenTrainer}


{
//////////////////////////////////////////////////////////////////////
// function TrainKohonen
//
// StgAlias : alias of trainer object
// StgData : alias of training data
// IntEpochs : number of epochs to train for
// FltLearningRate : learning rate parameter
// DyaIntNeighbourhood : one dimensional variant array of ints 
//                       specifying the neighbourhood for each dimension
//
//////////////////////////////////////////////////////////////////////
}

function TrainKohonen ( StgAlias : AnsiString; StgData : AnsiString; IntEpochs : integer; FltLearningRate : real; DyaIntNeighbourhood : Variant ) : integer;

var

  StgCommand : AnsiString;
  IntLowBound : integer;
  IntHighBound : integer;
  StgNeighbourhood : AnsiString;
  IntCount : integer;

begin

  {check the array is one dimensional}
  if VarArrayDimCount ( DyaIntNeighbourhood ) <> 1 then
    raise Exception.Create('One-dimensional variant array expected');

  {convert the array into a string}
  IntLowBound := VarArrayLowBound ( DyaIntNeighbourhood, 1 );
  IntHighBound := VarArrayHighBound ( DyaIntNeighbourhood, 1 );

  StgNeighbourhood := 'neighbourhood ' + LIST_START;
  for IntCount := IntLowBound to IntHighBound - 1 do
  begin
    StgNeighbourhood := StgNeighbourhood + IntToStr ( DyaIntNeighbourhood [ IntCount ] );
    StgNeighbourhood := StgNeighbourhood + ITEM_DELIM;
    StgNeighbourhood := StgNeighbourhood + ' ';
  end; {for}
  StgNeighbourhood := StgNeighbourhood + IntToStr ( DyaIntNeighbourhood [ IntHighBound ] );
  StgNeighbourhood := StgNeighbourhood + LIST_END;
      
  StgCommand := 'train alias ' + StgAlias + ' data ' + StgData + ' epochs ' + IntToStr ( IntEpochs );
  StgCommand := StgCommand + ' LearningRate ' + FloatToStr ( FltLearningRate ) + ' ';
  StgCommand := StgCommand + StgNeighbourhood;

  TrainKohonen := fcSendCommand ( StgCommand );

end; {TrainKohonen}


{
//////////////////////////////////////////////////////////////////////
// function MapKohonen
//
// creates a coordinate-label map for a Kohonen network
//
// StgNetwork : alias of the network to map
// StgInputs : alias of the labelled map data
// StgOutputs : alias to assign the label-coordinate map data object to
//
//////////////////////////////////////////////////////////////////////
}

function MapKohonen ( StgNetwork : AnsiString; StgInput : AnsiString; StgOutput : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'map network ' + StgNetwork + ' input ' + StgInput + ' output ' + StgOutput;
  MapKohonen := fcSendCommand ( StgCommand );

end; {MapKohonen}


{
//////////////////////////////////////////////////////////////////////
// function LabelKohonen
//
// creates a list of labels for input vectors based upon the coordinate
// map generated by a Map operation and a coordinate list generated by
// a recall operation
//
// StgNetwork : alias of the Kohonen network 
// StgMap : alias of the Map data object
// StgCoord : alias of the coordinates data object
// StgOutput : alias to assign the output labels to
//
//////////////////////////////////////////////////////////////////////
}

function LabelKohonen ( StgNetwork : AnsiString; StgMap : AnsiString; StgCoord : AnsiString; StgOutput : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'label network ' + StgNetwork + ' map ' + StgMap;
  StgCommand := StgCommand + ' coord ' + StgCoord + ' output ' + StgOutput;
  LabelKohonen := fcSendCommand ( StgCommand );

end; {LabelKohonen}


{
//////////////////////////////////////////////////////////////////////
// function SaveKohonen
//
// saves a Kohonen to file
//
// StgAlias : alias of Kohonen to save
// StgFileName : file to save to
//
//////////////////////////////////////////////////////////////////////
}

function SaveKohonen ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'save Kohonen alias ' + StgAlias + ' filename ' + START_QUOTE + StgFileName + END_QUOTE;
  SaveKohonen := fcSendCommand ( StgCommand );

end; {SaveKohonen}


{
//////////////////////////////////////////////////////////////////////
// function LoadKohonen
//
// load a Kohonen from file
//
// StgAlias : alias to assign Kohonen to
// StgFileName : name of file to load
//
//////////////////////////////////////////////////////////////////////
}

function LoadKohonen ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'load Kohonen alias ' + StgAlias + ' filename ' + START_QUOTE + StgFileName + END_QUOTE;
  LoadKohonen := fcSendCommand ( StgCommand );

end; {LoadKohonen}


{
//////////////////////////////////////////////////////////////////////
// General ANN Functions
//////////////////////////////////////////////////////////////////////
}


{
//////////////////////////////////////////////////////////////////////
// function CreateBPTrainer
//
// creates a trainer object. Works for MLPs and
// FuNNs
//
// StgAlias = alias to assign the trainer to
// StgNetwork = alias of the network to train
//
//////////////////////////////////////////////////////////////////////
}

function CreateBPTrainer ( StgAlias : AnsiString; StgNetwork : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  {assemble and send the command string}
  StgCommand := 'create BPTrainer alias ' + StgAlias + ' network ' + StgNetwork;
  CreateBPTrainer := fcSendCommand ( StgCommand );

end; {CreateBPTrainer}


{
//////////////////////////////////////////////////////////////////////
// function CreateGATrainer
//
// creates a GA trainer object. Works for MLPs 
// and FuNNs
//
// StgAlias = alias to assign the trainer to
// StgNetwork = alias of the network to train
// FltMinWeight = minimum weight value
// FltMaxWeight = maximum weight value
// IntPopSize = population size
//
//////////////////////////////////////////////////////////////////////
}

function CreateGATrainer ( StgAlias : AnsiString; StgNetwork : AnsiString; FltMinweight : real; FltMaxWeight : real; IntPopSize : integer ) : integer;

var

  StgCommand : AnsiString;
  StgMinWeight : AnsiString;
  StgMaxWeight : AnsiString;
  StgPopSize : AnsiString;

begin

  {convert the parameters to strings}
  StgMinWeight := FloatToStr ( FltMinWeight );
  StgMaxWeight := FloatToStr ( FltMaxWeight );
  StgPopSize := IntToStr ( IntPopSize );

  {assemble the command string}
  StgCommand := 'create GATrainer alias ' + StgAlias + ' network ' + StgNetwork;
  StgCommand := StgCommand + ' minweight ' + StgMinWeight + ' maxweight ' + StgMaxWeight;
  StgCommand := StgCommand + ' population ' + StgPopSize;

  {send the command string}
  CreateGATrainer := fcSendCommand ( StgCommand );

end; {CreateGATrainer}


{
//////////////////////////////////////////////////////////////////////
// function BPTrainANN
//
// BP trains an ANN 
//
// StgAlias = the alias of the trainer
// StgDataAlias = alias of the training data
// IntEpochs = epochs to train for
// FltLearningRate = learning rate
// FltMomentum = momentum term
// FltTerminating = terminating RMS
// BlnBatch true = batch mode, false = pattern
//
//////////////////////////////////////////////////////////////////////
}

function BPTrainANN( StgAlias : AnsiString; StgDataAlias : AnsiString; IntEpochs : integer; FltLearningRate : real; FltMomentum : real; FltTerminating : real; BlnBatch : boolean ) : integer;

var

  StgCommand : AnsiString;
  StgEpochs : AnsiString;
  StgLearningRate : AnsiString;
  StgMomentum : AnsiString;
  StgTerminating : AnsiString;
  StgMode : AnsiString;

begin

  {convert the paraemters to strings}
  StgEpochs := IntToStr ( IntEpochs );
  StgLearningRate := FloatToStr ( FltLearningRate );
  StgMomentum := FloatToStr ( FltMomentum );
  StgTerminating := FloatToStr ( FltTerminating );
  if BlnBatch then
    StgMode := 'batch'
  else
    StgMode := 'pattern';

  {assemble the command string}
  StgCommand := 'train alias ' + StgAlias + ' data ' + StgDataAlias + ' epochs ' + StgEpochs + ' LearningRate ' + StgLearningRate;
  StgCommand := StgCommand + ' Momentum ' + StgMomentum + ' terminating ' + StgTerminating + ' mode ' + StgMode;
  
  {send the command}
  BPTrainANN := fcSendCommand ( StgCommand );

end; {BPTrainANN}


{
//////////////////////////////////////////////////////////////////////
// function GATrainANN
//
// GA trains a network. Works for MLPs and FuNNs
//
// StgAlias = alias of the trainer object
// StgData = alias of training data object
// IntGenerations = number of generations
// FltMutation = mutation rate
// IntSelection = selection strategy 
// BlnNormalisation = fitness normalisation 
// FltTerminating = terminating RMS
//
//////////////////////////////////////////////////////////////////////
}

function GATrainANN ( StgAlias : AnsiString; StgData : AnsiString; IntGenerations : integer; FltMutation : real; IntSelection : integer; IntCrossover : integer; BlnElitism : boolean; BlnNormalisation : boolean; FltTerminating : real ) : integer;

var

  StgCommand : AnsiString;
  StgGenerations : AnsiString;
  StgMutation : AnsiString;
  StgSelection : AnsiString;
  StgElitism : AnsiString;
  StgNormalisation : AnsiString;
  StgTerminating : AnsiString;
  StgCrossover : AnsiString;
  
begin

  {convert the parameters to strings}
  StgGenerations := IntToStr ( IntGenerations );
  StgMutation := FloatToStr ( FltMutation );
  StgCrossover := IntToSTr ( IntCrossover );
  if IntSelection = TOURNAMENT then
    StgSelection := 'Tournament'
  else
    StgSelection := 'Roulette';

  if BlnElitism then
    StgElitism := 'on'
  else
    StgElitism := 'off';

  if BlnNormalisation then
    StgNormalisation := 'on'
  else
    StgNormalisation := 'off';

  StgTerminating := FloatToStr ( FltTerminating );

  {assemble the command string}
  StgCommand := 'train alias ' + StgAlias + ' data ' + StgData + ' generations ' + StgGenerations;
  StgCommand := StgCommand + ' mutation ' + StgMutation + ' selection ' + StgSelection ;
  StgCommand := StgCommand + ' crossover ' + StgCrossover;
  StgCommand := StgCommand + ' elitism ' + StgElitism + ' normalisation ' + StgNormalisation ;
  StgCommand := StgCommand + ' terminating ' + StgTerminating;
  
  {finished assembling the command, so send it}
  GATrainANN := fcSendCommand ( StgCommand );

end; {GATrainANN}


{
//////////////////////////////////////////////////////////////////////
// function LWFTrainANN
//
// trains an MLP or FuNN network using Learning with forgetting
//
// StgAlias : alias of training object
// StgDataAlias : alias of training data
// IntEpochs : number of epochs to train for
// FltLearningRate : learning rate term
// FltMomentum : momentum
// FltDecay : weight decay term
// FltTerminating : terminating RMS error
// BlnBatch : flag to indicate batch mode training
//
//////////////////////////////////////////////////////////////////////
}
function LWFTrainANN( StgAlias : AnsiString; StgDataAlias : AnsiString; IntEpochs : integer; FltLearningRate : real; FltMomentum : real; FltDecay : real; FltTerminating : real; BlnBatch : boolean ) : integer;

var

  StgCommand : AnsiString;
  StgEpochs : AnsiString;
  StgLearningRate : AnsiString;
  StgMomentum : AnsiString;
  StgDecay : AnsiString;
  StgTerminating : AnsiString;
  StgMode : AnsiString;

begin

  {convert the paraemters to strings}
  StgEpochs := IntToStr ( IntEpochs );
  StgLearningRate := FloatToStr ( FltLearningRate );
  StgMomentum := FloatToStr ( FltMomentum );
  StgDecay := FloatToStr ( FltDecay );
  StgTerminating := FloatToStr ( FltTerminating );
  if BlnBatch then
    StgMode := 'batch'
  else
    StgMode := 'pattern';

  {assemble the command string}
  StgCommand := 'train alias ' + StgAlias + ' data ' + StgDataAlias + ' epochs ' + StgEpochs + ' LearningRate ' + StgLearningRate;
  StgCommand := StgCommand+ ' Momentum ' + StgMomentum + ' decay ' + StgDecay + ' terminating ' + StgTerminating + ' mode ' + StgMode;
  
  {send the command}
  LWFTrainANN := fcSendCommand ( StgCommand );

end; {LWFTrainANN}


{
//////////////////////////////////////////////////////////////////////
// function Recall
//
// recalls the specified ANN using the input
// dataset specified, writing the results to the
// specified output
//
// StgAlias = alias of MLP to recall
// StgInput = alias of input data set
// StgOutput = alias to assign output data to
//
//////////////////////////////////////////////////////////////////////
}

function Recall ( StgAlias : AnsiString; StgInput : AnsiString; StgOutput : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'recall network ' + StgAlias + ' input ' + StgInput + ' output ' + StgOutput;
  Recall := fcSendCommand ( StgCommand );

end; {Recall}


{
//////////////////////////////////////////////////////////////////////
// function GetTrained
//
// returns a trained network froma GA trainer object or a FuNN LWF
// object
//
//////////////////////////////////////////////////////////////////////
}

function GetTrained ( StgSource : AnsiString; StgDestination : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'gettrained source ' + StgSource + ' destination ' + StgDestination;
  GetTrained := fcSendCommand ( StgCommand );

end; {GetTrained}


{
//////////////////////////////////////////////////////////////////////
// Data Object Functions
//////////////////////////////////////////////////////////////////////
}


{
//////////////////////////////////////////////////////////////////////
// function CreateData
//
// creates a data object of the specified 
// dimensions
//
// StgAlias = alias object is assigned to
// IntInputs = inputs in the data set
// IntOutputs = outputs in the data set
// IntRows = data rows in the object
// BlnLabels = turns labelling on or off
//
//////////////////////////////////////////////////////////////////////
}

function CreateData ( StgAlias : AnsiString; IntInputs : integer; IntOutputs : integer; IntRows : integer; BlnLabels : boolean ) : integer;

var

  StgCommand : AnsiString;
  StgInputs : AnsiString;
  StgOutputs : AnsiString; 
  StgRows : AnsiString;
  StgLabels : AnsiString;

begin

  {convert the arguments to strings}
  StgInputs := IntToStr ( IntInputs );
  StgOutputs := IntToStr ( IntOutputs );
  StgRows := IntToStr ( IntRows );
  if BlnLabels then
    StgLabels := 'on'
  else
    StgLabels := 'off';

  {assemble the command string}
  StgCommand := 'create Data alias ' + StgAlias + ' inputs ' + StgInputs;
  StgCommand := StgCommand + ' outputs ' + StgOutputs + ' rows ' + StgRows;
  StgCommand := StgCommand + ' labels ' + StgLabels;

  {finished assembling the command string, so send it}
  CreateData := fcSendCommand ( StgCommand );

end; {CreateData}


{
//////////////////////////////////////////////////////////////////////
// function GetDataStructure
//
// returns the structure of the specified data object
//
// StgAlias : alias of the data object
// IntInputs : number of inputs
// IntOutputs : number of outputs
// IntRows : number of storage rows
// IntDataRows : number of rows of data in the object
// BlnLabels : whether or not labels are active in the object
//
//////////////////////////////////////////////////////////////////////
}

function GetDataStructure ( StgAlias : AnsiString; var IntInputs : integer; var IntOutputs : integer; var IntRows : integer; var IntDataRows : integer; var BlnLabels : boolean ) : integer;

var

  StgCommand : AnsiString;
  IntReturn : integer;
  StgResult : AnsiString;
  StgPResult : PChar;
  IntResultLength : integer;

begin

  StgCommand := 'query structureof ' + StgAlias;
  IntReturn := fcSendCommand ( StgCommand );
  if IntReturn >= 0 then
  begin
    StgResult := fcGetResultString;
    {convert to a C string}
    IntResultLength := length ( StgResult ) + 1;
    GetMem ( StgPResult, IntResultLength );
    StrPCopyWorks ( StgPResult, StgResult );
    {parse the C string}
    {check it is a data object}
    if TagPresent ( StgPResult, 'datarows' ) then
    begin 
      {parse the results string}
      IntReturn := GetArgumentInt ( StgPResult, 'inputs', IntInputs );
      if IntReturn <> SUCCESS then
        raise Exception.Create('Internal parsing error');
  
      IntReturn := GetArgumentInt ( StgPResult, 'outputs', IntOutputs );
      if IntReturn <> SUCCESS then
        raise Exception.Create('Internal parsing error');
      
      IntReturn := GetArgumentInt ( StgPResult, 'rows', IntRows );
      if IntReturn <> SUCCESS then
        raise Exception.Create('Internal parsing error');

      IntReturn := GetArgumentInt ( StgPResult, 'datarows', IntDataRows );
      if IntReturn <> SUCCESS then
        raise Exception.Create('Internal parsing error');
              
      IntReturn := GetArgumentBln ( StgPResult, 'labels', BlnLabels );
      if IntReturn <> SUCCESS then
        raise Exception.Create('Internal parsing error');
    end {if}  
    else
      IntReturn := INCORRECT_OBJECT_TYPE;
    FreeMem ( StgPResult );
  end; {if}

  GetDataStructure := IntReturn;

end; {GetDataStructure}


{
//////////////////////////////////////////////////////////////////////
// function AppendRow
//
// appends a row of data to a data object
//
// StgAlias : alias of data object to append data to
// DyaFltInputs : one dimensional variant array of floats containing 
//                the inputs data
// DyaFltOutputs : one dimensional variant array of floats containing
//                 the outputs data
//
//////////////////////////////////////////////////////////////////////
}

function AppendRow ( StgAlias : AnsiString; DyaFltInputs : Variant; DyaFltOutputs : Variant; DyaStgLabels : Variant; DyaFltLabelWeights : Variant; BlnInputs : boolean; BlnOutputs : boolean; BlnLabels : boolean ) : integer;

var

  StgCommand : AnsiString;
  IntCount : integer;
  IntInputLow : integer;
  IntInputHigh : integer;
  IntOutputLow : integer;
  IntOutputHigh : integer;
  IntLabelLow : integer;
  IntLabelHigh : integer;
  IntLabelWeightLow : integer;
  IntLabelWeightHigh : integer;

begin

  {throw an exception if the arrays are the wrong dimension}
  if BlnInputs then
  begin
    if VarArrayDimCount ( DyaFltInputs ) <> 1 then
      raise Exception.Create('One-dimensional variant array expected');
  end;

  if BlnOutputs then
  begin
    if VarArrayDimCount ( DyaFltOutputs ) <> 1 then
      raise Exception.Create('One-dimensional variant array expected');
  end;

  if BlnLabels then
  begin
    if VarArrayDimCount ( DyaStgLabels ) <> 1 then
      raise Exception.Create('One-dimensional variant array expected');
    if VarArrayDimCount ( DyaFltLabelWeights ) <> 1 then
      raise Exception.Create('One-dimensional variant array expected');
  end; {if}

  {assemble the command string}
  StgCommand := 'append alias ' + StgAlias;

  {first the inputs}
  if BlnInputs then
  begin
    IntInputLow := VarArrayLowBound ( DyaFltInputs, 1 );
    IntInputHigh := VarArrayHighBound ( DyaFltInputs, 1 );

    StgCommand := StgCommand + ' inputs ' + LIST_START;
    for IntCount := IntInputLow to IntInputHigh - 1 do
    begin
      StgCommand := StgCommand + FloatToStr ( DyaFltInputs [ IntCount ] );
      StgCommand := StgCommand + ITEM_DELIM;
      StgCommand := StgCommand + ' ';
    end;
    StgCommand := StgCommand + FloatToStr ( DyaFltInputs [ IntInputHigh ] );
    StgCommand := StgCommand + LIST_END;
  end; {if}

  {then the outputs}
  if BlnOutputs then
  begin
    IntOutputLow := VarArrayLowBound ( DyaFltOutputs, 1 );
    IntOutputHigh := VarArrayHighBound ( DyaFltOutputs, 1 );

    StgCommand := StgCommand + ' outputs ' + LIST_START;
    for IntCount := IntOutputLow to IntOutputHigh - 1 do
    begin
      StgCommand := StgCommand + FloatToStr ( DyaFltOutputs [ IntCount ] );
      StgCommand := StgCommand + ITEM_DELIM;
      StgCommand := StgCommand + ' ';
    end;
    StgCommand := StgCommand + FloatToStr ( DyaFltOutputs [ IntOutputHigh ] );
    StgCommand := StgCommand + LIST_END;
  end; {if}

  {then the labels and label weights}
  if BlnLabels then
  begin
    IntLabelLow := VarArrayLowBound ( DyaStgLabels, 1 );
    IntLabelHigh := VarArrayHighBound ( DyaStgLabels, 1 );
    IntLabelWeightLow := VarArrayLowBound ( DyaStgLabels, 1 );
    IntLabelWeightHigh := VarArrayHighBound ( DyaStgLabels, 1 );

    StgCommand := StgCommand + ' labels ' + LIST_START;
    for IntCount := IntLabelLow to IntLabelHigh - 1 do
    begin
      StgCommand := StgCommand + DyaStgLabels [ IntCount ];
      StgCommand := StgCommand + ITEM_DELIM;
      StgCommand := StgCommand + ' ';
    end;
    StgCommand := StgCommand + DyaStgLabels [ IntLabelHigh ];
    StgCommand := StgCommand + LIST_END;

    StgCommand := StgCommand + ' labelweights ' + LIST_START;
    for IntCount := IntLabelWeightLow to IntLabelWeightHigh - 1 do
    begin
      StgCommand := StgCommand + FloatToStr ( DyaFltLabelWeights [ IntCount ] );
      StgCommand := StgCommand + ITEM_DELIM;
      StgCommand := StgCommand + ' ';
    end;
    StgCommand := StgCommand + FloatToStr ( DyaFltLabelWeights [ IntLabelWeightHigh ] );
    StgCommand := StgCommand + LIST_END;

  end; {if BlnLabels}


  {send the completed command string}
  AppendRow := fcSendCommand ( StgCommand );

end; {AppendRow}


{
//////////////////////////////////////////////////////////////////////
// function PurgeData
//
// purges all data from a data object
//
// StgAlias : alias of data object
//
//////////////////////////////////////////////////////////////////////
}

function PurgeData ( StgAlias : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'purgedata alias ' + StgAlias;
  PurgeData := fcSendCommand ( StgCommand );

end; {PurgeData}


{
//////////////////////////////////////////////////////////////////////
// function ResetData
//
// resets the current row pointer of a data object to the first row in 
// the data set
//
// StgAlias : alias of the data object
//
//////////////////////////////////////////////////////////////////////
}

function ResetData ( StgAlias : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'resetdata alias ' + StgAlias;
  ResetData := fcSendCommand ( StgCommand );

end; {ResetData}


{
//////////////////////////////////////////////////////////////////////
// function GetNextRow
//
// returns the next row of the data set, advancing the current row
// pointer
//
// StgAlias : alias of data object
// DyaFltInputs : one dimensional variant array of floats that the
//                input data is returned into
// DyaFltOutputs : one dimensional variant array of floats that the
//                 output data is returned into
// DyaStgLabels : one dimensional variant array of strings that the
//                output data is returned into
// DyaFltLabelWeights : one dimensional variant array of floats that 
//                      the label weightings are returned into
// IntInputs : number of input columns
// IntOutputs : number of output columns
// IntNumLabels : number of labels attached to row
//
//////////////////////////////////////////////////////////////////////
}

function GetNextRow ( StgAlias : AnsiString; var DyaFltInputs : Variant; var DyaFltOutputs : Variant; var DyaStgLabels : Variant; var DyaFltLabelWeights : Variant; var IntInputs : integer; var IntOutputs : integer; var IntNumLabels : integer ) : integer;

var

  StgCommand : AnsiString;
  IntReturn : integer;
  StgResult : AnsiString;
  StgPResult : PChar;
  IntResultLength : integer;

begin

  StgCommand := 'query getnextrow ' + StgAlias;
  IntReturn := fcSendCommand ( StgCommand );
  if IntReturn >= 0 then
  begin
    StgResult := fcGetResultString;
    IntResultLength := length ( StgResult ) + 1;
    GetMem ( StgPresult, IntResultLength );
    StrPCopyWorks ( StgPResult, StgResult );
    if TagPresent ( StgPResult, 'inputs' ) then
    begin
      IntReturn := GetFltArguments ( StgPResult, 'inputs', DyaFltInputs, IntInputs );
      if IntReturn <> SUCCESS then
        raise Exception.Create('Internal parsing error');
    end {if}
    else
      IntInputs := 0;

    if TagPresent ( StgPResult, 'outputs' ) then
    begin
      IntReturn := GetFltArguments ( StgPResult, 'outputs', DyaFltOutputs, IntOutputs );
      if IntReturn <> SUCCESS then
        raise Exception.Create('Internal parsing error');
    end {if}
    else
      IntOutputs := 0;

    if TagPresent ( StgPResult, 'labels' ) then
    begin
      IntReturn := GetStgArguments ( StgPresult, 'labels', DyaStgLabels, IntNumLabels );
      if IntReturn <> SUCCESS then
        raise Exception.Create('Internal parsing error');
    end {if}
    else
      IntNumLabels := 0;

    if TagPresent ( StgPResult, 'labelweights' ) then
    begin
      IntReturn := GetFltArguments ( StgPResult, 'labelweights', DyaFltLabelWeights, IntNumLabels );
      if IntReturn <> SUCCESS then
        raise Exception.Create('Internal parsing error');
    end; {if}  

    FreeMem ( StgPResult );

  end; {if}

  GetNextRow := IntReturn;

end; {GetNextRow}


{
//////////////////////////////////////////////////////////////////////
// function SaveData
//
// saves the specified data object to disk
//
// StgAlias = alias of data object to save
// StgFileName = file to save to
//
//////////////////////////////////////////////////////////////////////
}

function SaveData ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  {assemble the command string and send it}
  StgCommand := 'save Data alias ' + StgAlias + ' filename ' + START_QUOTE + StgFilename + END_QUOTE;
  SaveData := fcSendCommand ( StgCommand );

end; {SaveData}


{
//////////////////////////////////////////////////////////////////////
// function LoadData
//
// loads a data object from disk
//
// StgAlias = alias to assign data object to
// StgFileName = name of file to load
//
//////////////////////////////////////////////////////////////////////
}

function LoadData ( StgAlias : AnsiString; StgFileName : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin
 
  {assemble and send the command string}
  StgCommand := 'load Data alias ' + StgAlias + ' filename ' + START_QUOTE + StgFileName + END_QUOTE;
  LoadData := fcSendCommand ( StgCommand );

end; {LoadData}


{
//////////////////////////////////////////////////////////////////////
// Data Transformation Functions
//////////////////////////////////////////////////////////////////////
}


{
//////////////////////////////////////////////////////////////////////
// function ShuffleData
//
// shuffles the data in the source object and
// assigns it to the destination object
//
// StgSource = alias of source object
// StgDestination = alias of destination object
//
//////////////////////////////////////////////////////////////////////
}

function ShuffleData ( StgSource : AnsiString; StgDestination : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  {assemble and send the command string}
  StgCommand := 'shuffle source ' + StgSource + ' destination ' + StgDestination ;
  ShuffleData := fcSendCommand ( StgCommand );

end; {ShuffleData}


{
//////////////////////////////////////////////////////////////////////
// function LinearNormaliseInputs
//
// linear normalises a data object using a 
// seperate max and min value for each column
//
// StgSource = alias of source data
// StgDestination = alias to assign destination
//  data to
// DyaFltMins = array of min values
// DyaFltMaxs = array of max values
//
//////////////////////////////////////////////////////////////////////
}

function LinearNormaliseInputs ( StgSource : AnsiString; StgDestination : AnsiString; DyaFltMins : Variant; DyaFltMaxs : Variant ) : integer;

var

  StgCommand : AnsiString;
  IntCount : integer;
  IntMinLow : integer;
  IntMinHigh : integer;
  IntMaxLow : integer;
  IntMaxHigh : integer;

begin

  {throw an exception if the arrays are the wrong dimension}
  if VarArrayDimCount ( DyaFltMins ) <> 1 then
    raise Exception.Create('One-dimensional variant array expected');
  if VarArrayDimCount ( DyaFltMaxs ) <> 1 then
    raise Exception.Create('One-dimensional variant array expected');
  
  {retrieve the bounds of the two arrays}
  IntMinLow := VarArrayLowBound ( DyaFltMins, 1 );
  IntMinHigh := VarArrayHighBound ( DyaFltMins, 1 );
  IntMaxLow := VarArrayLowBound ( DyaFltMaxs, 1 );
  IntMaxHigh := VarArrayHighBound ( DyaFltMaxs, 1 );

  {assemble the command string}
  StgCommand := 'normalise method linear source ' + StgSource + ' destination ' + StgDestination;
  StgCommand := StgCommand + ' inputcmaxs ' + LIST_START;

  for IntCount := IntMaxLow to IntMaxHigh - 1 do
  begin
    StgCommand := StgCommand  + FloatToStr ( DyaFltMaxs [ IntCount ] );
    StgCommand := StgCommand + ITEM_DELIM;
    StgCommand := StgCommand + ' ';
  end; {for}
  StgCommand := StgCommand + FloatToStr ( DyaFltMaxs [ IntmaxHigh ] );
  StgCommand := StgCommand + LIST_END;

  StgCommand := StgCommand + ' inputcmins ' + LIST_START;

  for IntCount := IntMinLow to IntMinHigh do
  begin
    StgCommand := StgCommand  + FloatToStr ( DyaFltMins [ IntCount ] );
    if IntCount <> IntMinHigh then
    begin
      StgCommand := StgCommand + ITEM_DELIM;
      StgCommand := StgCommand + ' ';
    end; {if}
  end; {for}
  StgCommand := StgCommand + LIST_END;

  {send the command}
  LinearNormaliseInputs := fcsendCommand ( StgCommand );

end; {LinearNormaliseInputs}


{
//////////////////////////////////////////////////////////////////////
// function LinearNormaliseOutputs
//
// linear normalises a data object using a 
// seperate max and min value for each column
//
// StgSource = alias of source data
// StgDestination = alias to assign destination
//  data to
// DyaFltMins = array of min values
// DyaFltMaxs = array of max values
//
//////////////////////////////////////////////////////////////////////
}

function LinearNormaliseOutputs ( StgSource : AnsiString; StgDestination : AnsiString; DyaFltMins : Variant; DyaFltMaxs : Variant ) : integer;

var

  StgCommand : AnsiString;
  IntCount : integer;
  IntMinLow : integer;
  IntMinHigh : integer;
  IntMaxLow : integer;
  IntMaxHigh : integer;

begin

  {throw an exception if the arrays are the wrong dimension}
  if VarArrayDimCount ( DyaFltMins ) <> 1 then
    raise Exception.Create('One-dimensional variant array expected');
  if VarArrayDimCount ( DyaFltMaxs ) <> 1 then
    raise Exception.Create('One-dimensional variant array expected');
  
  {retrieve the bounds of the two arrays}
  IntMinLow := VarArrayLowBound ( DyaFltMins, 1 );
  IntMinHigh := VarArrayHighBound ( DyaFltMins, 1 );
  IntMaxLow := VarArrayLowBound ( DyaFltMaxs, 1 );
  IntMaxHigh := VarArrayHighBound ( DyaFltMaxs, 1 );

  {assemble the command string}
  StgCommand := 'normalise method linear source ' + StgSource + ' destination ' + StgDestination;
  StgCommand := StgCommand + ' outputcmaxs ' + LIST_START;

  for IntCount := IntMaxLow to IntMaxHigh - 1 do
  begin
    StgCommand := StgCommand  + FloatToStr ( DyaFltMaxs [ IntCount ] );
    StgCommand := StgCommand + ITEM_DELIM;
    StgCommand := StgCommand + ' ';
  end; {for}
  StgCommand := StgCommand + FloatToStr ( DyaFltMaxs [ IntMaxHigh ] );
  StgCommand := StgCommand + LIST_END;

  StgCommand := StgCommand + ' outputcmins ' + LIST_START;

  for IntCount := IntMinLow to IntMinHigh - 1 do
  begin
    StgCommand := StgCommand  + FloatToStr ( DyaFltMins [ IntCount ] );
    StgCommand := StgCommand + ITEM_DELIM;
    StgCommand := StgCommand + ' ';
  end; {for}
  StgCommand := StgCommand + FloatToStr ( DyaFltMins [ IntMinHigh ] );
  StgCommand := StgCommand + LIST_END;

  {send the command}
  LinearNormaliseOutputs := fcsendCommand ( StgCommand );

end; {LinearNormaliseOutputs}


{
//////////////////////////////////////////////////////////////////////
// function LinearNormaliseAll
//
// linear normalises a data object using a 
// seperate max and min value for each column
//
// StgSource = alias of source data
// StgDestination = alias to assign destination
//  data to
// DyaFltMins = array of min values
// DyaFltMaxs = array of max values
//
//////////////////////////////////////////////////////////////////////
}

function LinearNormaliseAll ( StgSource : AnsiString; StgDestination : AnsiString; DyaFltInputMins : Variant; DyaFltInputMaxs : Variant; DyaFltOutputMins : Variant; DyaFltOutputMaxs : Variant ) : integer;

var

  StgCommand : AnsiString;
  IntCount : integer;
  IntInputMinLow : integer;
  IntInputMinHigh : integer;
  IntInputMaxLow : integer;
  IntInputMaxHigh : integer;
  IntOutputMinLow : integer;
  IntOutputMinHigh : integer;
  IntOutputMaxLow : integer;
  IntOutputMaxHigh : integer;

begin

  {throw an exception if the arrays are the wrong dimension}
  if VarArrayDimCount ( DyaFltInputMins ) <> 1 then
    raise Exception.Create('One-dimensional variant array expected');
  if VarArrayDimCount ( DyaFltInputMaxs ) <> 1 then
    raise Exception.Create('One-dimensional variant array expected');
  if VarArrayDimCount ( DyaFltOutputMins ) <> 1 then
    raise Exception.Create('One-dimensional variant array expected');
  if VarArrayDimCount ( DyaFltOutputMaxs ) <> 1 then
    raise Exception.Create('One-dimensional variant array expected');
  
  {retrieve the bounds of the two arrays}
  IntInputMinLow := VarArrayLowBound ( DyaFltInputMins, 1 );
  IntInputMinHigh := VarArrayHighBound ( DyaFltInputMins, 1 );
  IntInputMaxLow := VarArrayLowBound ( DyaFltInputMaxs, 1 );
  IntInputMaxHigh := VarArrayHighBound ( DyaFltInputMaxs, 1 );
  IntOutputMinLow := VarArrayLowBound ( DyaFltOutputMins, 1 );
  IntOutputMinHigh := VarArrayHighBound ( DyaFltOutputMins, 1 );
  IntOutputMaxLow := VarArrayLowBound ( DyaFltOutputMaxs, 1 );
  IntOutputMaxHigh := VarArrayHighBound ( DyaFltOutputMaxs, 1 );

  {assemble the command string}
  StgCommand := 'normalise method linear source ' + StgSource + ' destination ' + StgDestination;
  StgCommand := StgCommand + ' inputcmaxs ' + LIST_START;

  for IntCount := IntInputMaxLow to IntInputMaxHigh - 1 do
  begin
    StgCommand := StgCommand  + FloatToStr ( DyaFltInputMaxs [ IntCount ] );
    StgCommand := StgCommand + ITEM_DELIM;
    StgCommand := StgCommand + ' ';
  end; {for}
  StgCommand := StgCommand + FloatToStr ( DyaFltInputMaxs [ IntInputMaxHigh ] );
  StgCommand := StgCommand + LIST_END;

  StgCommand := StgCommand + ' inputcmins ' + LIST_START;

  for IntCount := IntInputMinLow to IntInputMinHigh - 1 do
  begin
    StgCommand := StgCommand  + FloatToStr ( DyaFltInputMins [ IntCount ] );
    StgCommand := StgCommand + ITEM_DELIM;
    StgCommand := StgCommand + ' ';
  end; {for}
  StgCommand := StgCommand + FloatToStr ( DyaFltInputMins [ IntInputMinHigh ] );
  StgCommand := StgCommand + LIST_END;

  StgCommand := StgCommand + ' outputcmaxs ' + LIST_START;

  for IntCount := IntOutputMaxLow to IntOutputMaxHigh - 1 do
  begin
    StgCommand := StgCommand  + FloatToStr ( DyaFltOutputMaxs [ IntCount ] );
    StgCommand := StgCommand + ITEM_DELIM;
    StgCommand := StgCommand + ' ';
  end; {for}
  StgCommand := StgCommand + FloatToStr ( DyaFltOutputMaxs [ IntOutputMaxHigh ] );
  StgCommand := StgCommand + LIST_END;

  StgCommand := StgCommand + ' outputcmins ' + LIST_START;

  for IntCount := IntOutputMinLow to IntOutputMinHigh - 1 do
  begin
    StgCommand := StgCommand  + FloatToStr ( DyaFltOutputMins [ IntCount ] );
    StgCommand := StgCommand + ITEM_DELIM;
    StgCommand := StgCommand + ' ';
  end; {for}
  StgCommand := StgCommand + FloatToStr ( DyaFltOutputMins [ IntOutputMinHigh ] );
  StgCommand := StgCommand + LIST_END;

  {send the command}
  LinearNormaliseAll := fcsendCommand ( StgCommand );

end; {LinearNormaliseAll}


{
//////////////////////////////////////////////////////////////////////
// Data Statistics Functions
//////////////////////////////////////////////////////////////////////
}

{
//////////////////////////////////////////////////////////////////////
// CreateDataStats
//
// creates a data stats object based on the data
// object provided
//
// StgAlias = alias to assign data stats object to
// StgData = alias of data object
//
//////////////////////////////////////////////////////////////////////
}

function CreateDataStats ( StgAlias : AnsiString; StgData : AnsiString ) : integer;

var 

  StgCommand : AnsiString;

begin

  {assemble and send the command string}
  StgCommand := 'create DataStats alias ' + StgAlias + ' source ' + StgData;
  CreateDataStats := fcSendCommand ( StgCommand );

end; {CreateDataStats}


{
//////////////////////////////////////////////////////////////////////
// function GetInputMaximums
//
// returns the maximum input values of a data set froma Data 
// Statistics object
//
// StgAlias : alias of the data statistics object
// DyaFltInputMax : one dimensional variant array of floats that the
//                  maximum input values are returned into
// IntColumns : number of input columns
//
//////////////////////////////////////////////////////////////////////
}

function GetInputMaximums ( StgAlias : AnsiString; var DyaFltInputMax : Variant; var IntColumns : integer ) : integer;

var

  StgCommand : AnsiString;
  StgResult : AnsiString;
  StgPResult : PChar;
  IntResultLength : integer;
  IntReturn : integer;

begin

  StgCommand := StgCommand + 'query getstats max alias ' + StgAlias + ' inputs true outputs false';
  IntReturn := fcSendCommand ( StgCommand );
  if IntReturn >= 0 then
  begin
    {retrieve and parse the result}
    StgResult := fcGetResultString;
    IntResultLength := length ( StgResult ) + 1;
    GetMem ( StgPResult, IntResultLength );
    StrPCopyWorks ( StgPResult, StgResult );
    IntReturn := GetFltArguments ( StgPResult, 'inputs', DyaFltInputMax, IntColumns );
    FreeMem ( StgPResult );
  end; {if}

  GetInputMaximums := IntReturn;

end; {GetInputMaximums}


{
//////////////////////////////////////////////////////////////////////
// function GetOutputMaximums
//
// returns the maximum output values of a data set from a Data 
// Statistics object
//
// StgAlias : alias of the data statistics object
// DyaFltOutputMax : one dimensional variant array of floats that the
//                   maximum output values are returned into
// IntColumns : number of output columns
//
//////////////////////////////////////////////////////////////////////
}

function GetOutputMaximums ( StgAlias : AnsiString; var DyaFltOutputMax : Variant; var IntColumns : integer ) : integer;

var

  StgCommand : AnsiString;
  StgResult : AnsiString;
  StgPResult : PChar;
  IntResultLength : integer;
  IntReturn : integer;

begin

  StgCommand := StgCommand + 'query getstats max alias ' + StgAlias + ' inputs false outputs true';
  IntReturn := fcSendCommand ( StgCommand );
  if IntReturn >= 0 then
  begin
    {retrieve and parse the result}
    StgResult := fcGetResultString;
    IntResultLength := length ( StgResult ) + 1;
    GetMem ( StgPResult, IntResultLength );
    StrPCopyWorks ( StgPResult, StgResult );
    IntReturn := GetFltArguments ( StgPResult, 'outputs', DyaFltOutputMax, IntColumns );
    FreeMem ( StgPResult );
  end; {if}

  GetOutputMaximums := IntReturn;

end; {GetOutputMaximums}


{
//////////////////////////////////////////////////////////////////////
// function GetMaximums
//
// StgAlias : alias of the data statistics object
// DyaFltInputMax : one dimensional variant array of floats that the
//                  maximum input values are returned into
// IntInputColumns : number of input columns
// DyaFltOutputMax : one dimensional variant array of floats that the
//                   maximum output values are returned into
// IntOutputColumns : number of output columns
//
//////////////////////////////////////////////////////////////////////
}

function GetMaximums ( StgAlias : AnsiString; var DyaFltInputMax : Variant; var IntInputColumns : integer; var DyaFltOutputMax : Variant; var IntOutputColumns : integer ) : integer;

var

  StgCommand : AnsiString;
  StgResult : AnsiString;
  StgPResult :PChar;
  IntResultLength : integer;
  IntReturn : integer;

begin

  StgCommand := StgCommand + 'query getstats max alias ' + StgAlias + ' inputs true outputs true';
  IntReturn := fcSendCommand ( StgCommand );
  if IntReturn >= 0 then
  begin
    {retrieve and parse the result}
    StgResult := fcGetResultString;
    IntResultLength := length ( StgResult ) + 1;
    GetMem ( StgPResult, IntResultLength );
    StrPCopyWorks ( StgPResult, StgResult );
    IntReturn := GetFltArguments ( StgPResult, 'inputs', DyaFltInputMax, IntInputColumns );
    if IntReturn = SUCCESS then
      IntReturn := GetFltArguments ( StgPResult, 'outputs', DyaFltOutputMax, IntOutputColumns );
    FreeMem ( StgPResult );

  end; {if}

  GetMaximums := IntReturn;

end; {GetMaximums}


{
//////////////////////////////////////////////////////////////////////
// function GetInputMinimums
//
// returns the Minimum input values of a data set froma Data 
// Statistics object
//
// StgAlias : alias of the data statistics object
// DyaFltInputMin : one dimensional variant array of floats that the
//                  Minimum input values are returned into
// IntColumns : number of input columns
//
//////////////////////////////////////////////////////////////////////
}

function GetInputMinimums ( StgAlias : AnsiString; var DyaFltInputMin : Variant; var IntColumns : integer ) : integer;

var

  StgCommand : AnsiString;
  StgResult : AnsiString;
  StgPResult : PChar;
  IntResultLength : integer;
  IntReturn : integer;

begin

  StgCommand := StgCommand + 'query getstats min alias ' + StgAlias + ' inputs true outputs false';
  IntReturn := fcSendCommand ( StgCommand );
  if IntReturn >= 0 then
  begin
    {retrieve and parse the result}
    StgResult := fcGetResultString;
    IntResultLength := length ( StgResult ) + 1;
    GetMem ( StgPResult, IntResultLength );
    StrPCopyWorks ( StgPResult, StgResult );
    IntReturn := GetFltArguments ( StgPResult, 'inputs', DyaFltInputMin, IntColumns );
    FreeMem ( StgPResult );
  end; {if}

  GetInputMinimums := IntReturn;

end; {GetInputMinimums}


{
//////////////////////////////////////////////////////////////////////
// function GetOutputMinimums
//
// returns the Minimum output values of a data set from a Data 
// Statistics object
//
// StgAlias : alias of the data statistics object
// DyaFltOutputMin : one dimensional variant array of floats that the
//                   Minimum output values are returned into
// IntColumns : number of output columns
//
//////////////////////////////////////////////////////////////////////
}

function GetOutputMinimums ( StgAlias : AnsiString; var DyaFltOutputMin : Variant; var IntColumns : integer ) : integer;

var

  StgCommand : AnsiString;
  StgResult : AnsiString;
  StgPResult : PChar;
  IntResultLength : integer;
  IntReturn : integer;

begin

  StgCommand := StgCommand + 'query getstats min alias ' + StgAlias + ' inputs false outputs true';
  IntReturn := fcSendCommand ( StgCommand );
  if IntReturn >= 0 then
  begin
    {retrieve and parse the result}
    StgResult := fcGetResultString;
    IntResultLength := length (StgResult ) + 1;
    GetMem ( StgPresult, IntResultLength );
    StrPCopyWorks ( StgPResult, StgResult );
    IntReturn := GetFltArguments ( StgPResult, 'outputs', DyaFltOutputMin, IntColumns );
    FreeMem ( StgPResult );
  end; {if}

  GetOutputMinimums := IntReturn;

end; {GetOutputMinimums}


{
//////////////////////////////////////////////////////////////////////
// function GetMinimums
//
// StgAlias : alias of the data statistics object
// DyaFltInputMin : one dimensional variant array of floats that the
//                  Minimum input values are returned into
// IntInputColumns : number of input columns
// DyaFltOutputMin : one dimensional variant array of floats that the
//                   Minimum output values are returned into
// IntOutputColumns : number of output columns
//
//////////////////////////////////////////////////////////////////////
}

function GetMinimums ( StgAlias : AnsiString; var DyaFltInputMin : Variant; var IntInputColumns : integer; var DyaFltOutputMin : Variant; var IntOutputColumns : integer ) : integer;

var

  StgCommand : AnsiString;
  StgResult : AnsiString;
  StgPResult : PChar;
  IntResultLength : integer;
  IntReturn : integer;

begin

  StgCommand := StgCommand + 'query getstats min alias ' + StgAlias + ' inputs true outputs true';
  IntReturn := fcSendCommand ( StgCommand );
  if IntReturn >= 0 then
  begin
    {retrieve and parse the result}
    StgResult := fcGetResultString;
    IntResultLength := length ( StgResult );
    GetMem ( StgPResult, IntResultLength );
    StrPCopyWorks ( StgPResult, StgResult );
    IntReturn := GetFltArguments ( StgPResult, 'inputs', DyaFltInputMin, IntInputColumns );
    if IntReturn = SUCCESS then
      IntReturn := GetFltArguments ( StgPResult, 'outputs', DyaFltOutputMin, IntOutputColumns );
    FreeMem ( StgPResult );

  end; {if}

  GetMinimums := IntReturn;

end; {GetMinimums}


{
//////////////////////////////////////////////////////////////////////
// function GetInputAverages
//
// returns the Average input values of a data set froma Data 
// Statistics object
//
// StgAlias : alias of the data statistics object
// DyaFltInputMin : one dimensional variant array of floats that the
//                  Average input values are returned into
// IntColumns : number of input columns
//
//////////////////////////////////////////////////////////////////////
}

function GetInputAverages ( StgAlias : AnsiString; var DyaFltInputAverages : Variant; var IntColumns : integer ) : integer;

var

  StgCommand : AnsiString;
  StgResult : AnsiString;
  StgPResult : PChar;
  IntResultLength : integer;
  IntReturn : integer;

begin

  StgCommand := StgCommand + 'query getstats average alias ' + StgAlias + ' inputs true outputs false';
  IntReturn := fcSendCommand ( StgCommand );
  if IntReturn >= 0 then
  begin
    {retrieve and parse the result}
    StgResult := fcGetResultString;
    IntResultLength := length ( StgResult ) + 1;
    GetMem ( StgPResult, IntResultLength );
    StrPCopyWorks ( StgPResult, StgResult );
    IntReturn := GetFltArguments ( StgPResult, 'inputs', DyaFltInputAverages, IntColumns );
    FreeMem ( StgPResult );
  end; {if}

  GetInputAverages := IntReturn;

end; {GetInputAverages}


{
//////////////////////////////////////////////////////////////////////
// function GetOutputAverages
//
// returns the Average output values of a data set from a Data 
// Statistics object
//
// StgAlias : alias of the data statistics object
// DyaFltOutputMin : one dimensional variant array of floats that the
//                   Average output values are returned into
// IntColumns : number of output columns
//
//////////////////////////////////////////////////////////////////////
}

function GetOutputAverages ( StgAlias : AnsiString; var DyaFltOutputAverages : Variant; var IntColumns : integer ) : integer;

var

  StgCommand : AnsiString;
  StgResult : AnsiString;
  StgPResult : PChar;
  IntResultLength : integer;
  IntReturn : integer;

begin

  StgCommand := StgCommand + 'query getstats average alias ' + StgAlias + ' inputs false outputs true';
  IntReturn := fcSendCommand ( StgCommand );
  if IntReturn >= 0 then
  begin
    {retrieve and parse the result}
    StgResult := fcGetResultString;
    IntResultLength := length ( StgResult ) + 1;
    GetMem ( StgPResult, IntResultLength );
    StrPCopyWorks ( StgPResult, StgResult );
    IntReturn := GetFltArguments ( StgPResult, 'inputs', DyaFltOutputAverages, IntColumns );
    FreeMem ( StgPResult );
  end; {if}

  GetOutputAverages := IntReturn;

end; {GetOutputAverages}


{
//////////////////////////////////////////////////////////////////////
// function GetAverages
//
// StgAlias : alias of the data statistics object
// DyaFltInputMin : one dimensional variant array of floats that the
//                  Average input values are returned into
// IntInputColumns : number of input columns
// DyaFltOutputMin : one dimensional variant array of floats that the
//                   Average output values are returned into
// IntOutputColumns : number of output columns
//
//////////////////////////////////////////////////////////////////////
}
function GetAverages ( StgAlias : AnsiString; var DyaFltInputAverages : Variant; var IntInputColumns : integer; var DyaFltOutputAverages : Variant; var IntOutputColumns : integer ) : integer;

var

  StgCommand : AnsiString;
  StgResult : AnsiString;
  StgPResult : PChar;
  IntResultLength : integer;
  IntReturn : integer;

begin

  StgCommand := StgCommand + 'query getstats average alias ' + StgAlias + ' inputs true outputs true';
  IntReturn := fcSendCommand ( StgCommand );
  if IntReturn >= 0 then
  begin
    {retrieve and parse the result}
    StgResult := fcGetResultString;
    IntResultLength := length ( StgResult ) + 1;
    GetMem ( StgPResult, IntResultLength );
    StrPCopyWorks ( StgPResult, StgResult );
    IntReturn := GetFltArguments ( StgPResult, 'inputs', DyaFltInputAverages, IntInputColumns );
    if IntReturn = SUCCESS then
      IntReturn := GetFltArguments ( StgPResult, 'outputs', DyaFltOutputAverages, IntOutputColumns );
    FreeMem ( StgPResult );

  end; {if}

  GetAverages := IntReturn;

end; {GetAverages}


{
//////////////////////////////////////////////////////////////////////
// miscellaneous functions
//////////////////////////////////////////////////////////////////////
}


{
//////////////////////////////////////////////////////////////////////
// function StartLogging
//
// starts logging of the generated command strings
//
// StgFilename = name of log file
//
//////////////////////////////////////////////////////////////////////
}

function StartLogging ( StgFilename : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin
  
  StgCommand := StgCommand + 'log start filename (';
  StgCommand := StgCommand + StgFilename;
  StgCommand := StgCommand + ')';

  StartLogging := fcSendCommand ( StgCommand );

end; {StartLogging}
  

{
//////////////////////////////////////////////////////////////////////
// function StopLogging
//
// stops logging of command strings
//
//////////////////////////////////////////////////////////////////////
}

function StopLogging : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'log stop';
  StopLogging := fcSendCommand ( StgCommand );

end; {StopLogging}


{
//////////////////////////////////////////////////////////////////////
// function RunScript
//
// runs a script file
//
//////////////////////////////////////////////////////////////////////
}

function RunScript ( StgFilename : AnsiString ) : integer;

var

  StgCommand : AnsiString;

begin

  StgCommand := 'run filename (';
  StgCommand := StgCommand + StgFilename;
  StgCommand := StgCommand + ')';

  RunScript := fcSendCommand ( StgCommand );

end; {RunScript}


end. {FC3Lib.pas}
