{
////////////////////////////////////////////////////////////
// file ParseFunction.pas
//
// contains various string parsing functions 
//
// created on: 08/04/98
// created by: Mike Watts (mwatts@rann)
//
// last modified on: 11/04/98
// last modified by: Mike Watts (mwatts@rann)
////////////////////////////////////////////////////////////
}

unit ParseFunction;

interface
uses Dialogs, SysUtils;

{export declarations}
procedure StrPCopyWorks ( StgDestination : PChar; StgSource : AnsiString );
function StrRev ( StgSource : PChar ) : PChar;
function LTrim ( StgSource : PChar ) : PChar;
function RTrim ( StgSource : PChar ) : PChar;
function AllTrim ( StgSource : PChar ) : PChar;
function Pad ( StgSource : PChar ) : PChar;
function TokenToBool ( StgToken : PChar; var BlnToken : boolean ) : integer;
function TagPresent ( StgSource : PChar; StgTag : PChar ) : boolean;
function GetString ( StgSource : PChar; StgTag : PChar; var StgArgument : PChar ) : integer;
function GetCommand ( StgSource : PChar ) : PChar;
function GetArgumentStg ( StgSource : PChar; StgTag : Pchar; var StgArgument : PChar ) : integer;
function GetArgumentBln ( StgSource : PChar; StgTag : PChar; var BlnArgument : boolean ) : integer;
function GetArgumentInt ( StgSource : PChar; StgTag : PChar; var IntArgument : integer ) : integer;
function GetArgumentFlt ( StgSource : PChar; StgTag : PChar; var FltArgument : real ) : integer;
function ParseArray ( StgArray : PChar; var IntArgumentCount : integer ) : variant;
function GetStgArguments ( StgSource : PChar; StgTag : PChar; var DyaStgArguments : Variant; var IntArgumentCount : integer ) : integer;
function GetBlnArguments ( StgSource : PChar; StgTag : PChar; var DyaBlnArguments : Variant; var IntArgumentCount : integer ) : integer;
function GetIntArguments ( StgSource : PChar; StgTag : PChar; var DyaIntArguments : Variant; var IntArgumentCount : integer ) : integer;
function GetFltArguments ( StgSource : PChar; StgTag : PChar; var DyaFltArguments : Variant; var IntArgumentCount : integer ) : integer;

{various formatting constants}
const
  LIST_START : AnsiString = '{';
  LIST_END : AnsiString = '}';
  ITEM_DELIM : AnsiString = ',';
  START_QUOTE : AnsiString = '(';
  END_QUOTE : AnsiString = ')';
  SUCCESS : integer = 0;
  PARSE_NOT_FOUND : integer = -201;
  PARSE_UNKNOWN_ARGUMENT : integer = -202;
  DELIM_NOT_FOUND : integer = -203;
  INCORRECT_OBJECT_TYPE : integer = -204;
  INTERNAL_PARSE_ERROR : integer = -205;


implementation


{
////////////////////////////////////////////////////////////
// procedure StrPCopyWorks
//
// replaces the inbuilt StrPCopy functions, which doesn't
// seem to work with long strings
//
////////////////////////////////////////////////////////////
}

procedure StrPCopyWorks ( StgDestination : PChar; StgSource : AnsiString );

var

  Str_index : integer;
  Str_length : integer;
begin
  Str_length := length(StgSource);
  for Str_index:=1 to Str_length do
    StgDestination[Str_index-1]:=StgSource[Str_index];
  StgDestination[Str_length]:=#0;

end;

{
////////////////////////////////////////////////////////////
// function StrRev
//
// returns the source string reversed
//
////////////////////////////////////////////////////////////
}

function StrRev ( StgSource : PChar ) : PChar;

var

  IntLength : integer;
  IntNewLength : integer;
  StgReturn : PChar;
  IntCount : integer;
  IntDest : integer;
 
begin

  IntLength := StrLen ( StgSource );
  IntNewLength := IntLength + 1;
  GetMem ( StgReturn, IntNewLength );
  IntDest := 0;
  for IntCount := IntLength - 1 downto 0 do
  begin
    StgReturn [ IntDest ] := StgSource [ IntCount ];
    IntDest := IntDest + 1;
  end;

  {append the null terminator}
  StgReturn [ IntLength ] := #0;

  StrRev := StgReturn;

end; {StrRev}
 

{
////////////////////////////////////////////////////////////
// function LTrim
//
// returns a string stripped of leading spaces
//
////////////////////////////////////////////////////////////
}

function LTrim ( StgSource : PChar ) : PChar;

var
  
  IntSourceLength : integer;
  IntCurr : integer;
  IntCount : integer;
  IntNewLength : integer;
  StgReturn : PChar; 
  IntDest : integer;

begin

  IntSourceLength := StrLen ( StgSource );
  IntCurr := 0;
  IntCount := 0;

  while ( IntCurr < IntSourceLength ) and ( StgSource [ IntCurr ] = ' ' ) do
  begin
    IntCurr := IntCurr + 1;
    IntCount := IntCount + 1;
  end; {while}

  IntNewLength := IntSourceLength - IntCount;
  IntNewLength := IntNewLength + 1;

  GetMem ( StgReturn, IntNewLength );

  IntCurr := 0;
  while ( IntCurr < IntSourceLength ) and ( StgSource [ IntCurr ] = ' ' ) do
  begin
    IntCurr := IntCurr + 1;
  end; {while}

  IntDest := 0;
  while  StgSource [ IntCurr ] <> #0 do
  begin
    StgReturn [ IntDest ] := StgSource [ IntCurr ];
    IntDest := IntDest + 1;
    IntCurr := IntCurr + 1;
  end; {while}

  {add the null terminator}
  StgReturn [ IntDest ] := #0;

  LTrim := StgReturn;

end; {LTrim}


{
////////////////////////////////////////////////////////////
// function RTrim
// 
// returns a string stripped of trailing spaces
//
////////////////////////////////////////////////////////////
}

function RTrim ( StgSource : PChar ) : PChar;

var

  StgRev : PChar;
  StgTrimmed : PChar;
  StgReturn : PChar;

begin

  {reverse the string}
  StgRev := StrRev ( StgSource );

  {LTrim it}
  StgTrimmed := LTrim ( StgRev );

  {reverse it again}
  StgReturn := StrRev ( StgTrimmed );

  {garbage collection}
  FreeMem ( StgRev );
  FreeMem ( StgTrimmed );

  RTrim := StgReturn;

end; {RTrim}


{
////////////////////////////////////////////////////////////
// function Trim
//
// returns a string stripped of leading and trailing spaces
//
////////////////////////////////////////////////////////////
}

function AllTrim ( StgSource : PChar ) : PChar;

var

  StgLTrimmed : PChar;
  StgReturn : PChar;

begin

  StgLTrimmed := LTrim ( StgSource );
  StgReturn := RTrim ( StgLTrimmed );

  {garbage collection}
  FreeMem ( StgLTrimmed );

  AllTrim := StgReturn;

end; {AllTrim}


{
////////////////////////////////////////////////////////////
// function Pad
//
// returns a string with a space added to the front and end
// of StgSource
//
////////////////////////////////////////////////////////////
}

function Pad ( StgSource : PChar ) : PChar;

var

 IntSourceLength : integer;
 IntNewLength : integer;
 StgReturn : PChar;
 IntCount : integer;

begin

  IntSourceLength := StrLen ( StgSource );
  IntNewLength := IntSourceLength + 3;
  GetMem ( StgReturn, IntNewLength );
  StgReturn [ 0 ] := ' ';
  
  for IntCount := 0 to IntSourceLength -1 do
  begin
    StgReturn [ IntCount + 1 ] := StgSource [ IntCount ];
  end;

  {add the trailing space}
  StgReturn [ IntNewLength - 2 ] := ' ';
  {add the null character}
  StgReturn [ IntNewLength - 1 ] := #0;

  Pad := StgReturn;

end;


{
////////////////////////////////////////////////////////////
// function TokenToBool
//
// converts the token StgToken into the equivalent boolean
// value. Regards "true" and "on" as true and "false" and
// "off" as false.
//
// returns PARSE_UNKNOWN_ARGUMENT if the token is unknown
// returns SUCCESS otherwise
//
////////////////////////////////////////////////////////////
}

function TokenToBool ( StgToken : PChar; var BlnToken : boolean ) : integer;

var

  StgArgument : PChar;
  IntReturn : integer;

begin

  {strip the token and initliase the vars}
  StgArgument := AllTrim ( StgToken );
  IntReturn := SUCCESS;

  {convert the string to a bool}
  if  StrComp ( 'on', StgArgument ) = 0 then
    BlnToken := true
  else if StrComp ( 'true', StgArgument ) = 0 then
    BlnToken := true
  else if StrComp ( 'off', StgArgument ) = 0 then
    BlnToken := false
  else if StrComp ( 'false', StgArgument ) = 0 then
    BlnToken := false
  else
    IntReturn := PARSE_UNKNOWN_ARGUMENT;

  {garbage collection}
  FreeMem ( StgArgument );

  TokenToBool := IntReturn;

end; {TokenToBool}


{
////////////////////////////////////////////////////////////
// function TagPresent
//
// returns true if the string StgTag is present in StgSource
// returns false otherwise
//
////////////////////////////////////////////////////////////
}

function TagPresent ( StgSource : PChar; StgTag : PChar ) : boolean;

var

  PtrChrReturn : PChar;

begin

  PtrChrReturn := StrPos ( StgSource, StgTag );
  if PtrChrReturn = nil then
    TagPresent := false
  else
    TagPresent := true;

end; {TagPresent}


{
///////////////////////////////////////////////////////////
// function GetString
//
// returns the quoted string following the tag StgTag.
//
////////////////////////////////////////////////////////////
}

function GetString ( StgSource : PChar; StgTag : PChar; var StgArgument : PChar ) : integer;

var

  StgTrimTag : PChar;
  StgUseTag : PChar;
  StgCommandString : PChar;
  IntReturn : integer;
  StgTagPos : PChar;
  IntSkip : integer;
  StgEndTag : PChar;
  IntCount : integer;
  BlnFound : boolean;
  IntStrLength : integer;
  IntStart : integer;
  IntLength : integer;
  IntIndex : integer;
  
begin

  StgTrimTag := AllTrim ( StgTag );
  StgUseTag := Pad ( StgTrimTag );
  IntReturn := SUCCESS;
  StgCommandString := Pad ( StgSource );

  if not TagPresent ( StgCommandString, StgUseTag ) then
    IntReturn := PARSE_NOT_FOUND
  else
  begin
    {find the tag}
    StgTagPos := StrPos ( StgCommandString, StgUseTag );

    {skip the tag portion}
    IntSkip := StrLen ( StgUseTag ) - 1;
    StgEndTag := @StgTagPos [ IntSkip ];

    {find the start quote}
    IntCount := 0;
    BlnFound := false;
    IntStrLength := StrLen ( StgEndTag );
    while ( not BlnFound ) and ( IntCount < IntStrLength - 1 ) do
    begin
      if StgEndTag [ IntCount ] = START_QUOTE then
        BlnFound := true
      else
        IntCount := IntCount + 1;
    end; {while}

    if not BlnFound then
    begin
      IntReturn := DELIM_NOT_FOUND;
    end {if}
    else
    begin
      {get the length of the argument}
      IntStart := IntCount + 1;
      IntLength := 0;
      BlnFound := false;
      while ( not BlnFound ) and ( IntCount < IntStrLength ) do
      begin
        if StgEndTag [ IntCount ] = END_QUOTE then
          BlnFound := true
        else
        begin
          IntLength := IntLength + 1;
          IntCount := IntCount + 1;
        end; {if-else}
      end; {while}
      if not BlnFound then
      begin
        IntReturn := DELIM_NOT_FOUND;
      end {if}
      else
      begin
        {create the string}
        GetMem ( StgArgument, IntLength );
        IntCount := IntStart;
        IntIndex := 0;
        BlnFound := false;
        while ( not BlnFound ) and ( IntCount < IntStrLength ) do
        begin
          if StgEndTag [ IntCount ] = END_QUOTE then
            BlnFound := true
          else
          begin
            StgArgument [ IntIndex ] := StgEndTag [ IntCount ];
            IntIndex := IntIndex + 1;
            IntCount := IntCount + 1;
          end; {else}
        end; {while}

        if not BlnFound then
        begin
          IntReturn := DELIM_NOT_FOUND;
        end
        else
        begin
          {append the null character}
          StgArgument [ IntIndex ] := #0;
        end; {else}
      end; {else}
    end; {else}

  end; {else}

  {garbage collection}
  FreeMem ( StgCommandString );
  FreeMem ( StgUseTag );

  GetString := IntReturn;

end; {GetString}


{
////////////////////////////////////////////////////////////
// function GetCommand
//
// returns the first complete string in the command
//
////////////////////////////////////////////////////////////
}

function GetCommand ( StgSource : PChar ) : PChar;

var

  StgTrimmed : PChar;
  IntLength : integer;
  StgReturn : PChar;
  IntCount : integer;
  
begin
  
  {trim the leading spaces}
  StgTrimmed := AllTrim ( StgSource );
  IntLength := StrLen ( StgTrimmed ) + 1;
  GetMem ( StgReturn, IntLength );
   
  {copy over the chars until a space is found}
  IntCount := 0;
  while ( IntCount < IntLength - 1 ) and ( StgTrimmed [ IntCount ] <> ' ' ) do
  begin
    StgReturn [ IntCount ] := StgTrimmed [ IntCount ];
    IntCount := IntCount + 1;
  end; {while}

  {append the null character}
  StgReturn [ IntCount ] := #0;

  {garbage collection}
  FreeMem ( StgTrimmed );

  GetCommand := StgReturn;

end; {GetCommand}


{
////////////////////////////////////////////////////////////
// function GetArgumentStg
//
// returns the string argument following the tag StgTag
// into StgArgument.
//
// returns PARSE_NOT_FOUND if StgTag is not present in 
//  StgSource
//
////////////////////////////////////////////////////////////
}

function GetArgumentStg ( StgSource : PChar; StgTag : Pchar; var StgArgument : PChar ) : integer;

var

  StgTrimmed : PChar;
  StgUseTag : PChar;
  IntReturn : integer;
  StgCommandString : PChar;
  StgTagPos : PChar;
  IntSkip : integer;
  StgEndTag : PChar;
  IntCount : integer;
  IntIndex : integer;
  
begin

  StgTrimmed := AllTrim ( StgTag );
  StgUseTag := Pad ( StgTrimmed );
  IntReturn := SUCCESS;
  StgCommandString := Pad ( StgSource );
  
  if not TagPresent ( StgCommandString, StgUseTag ) then
  begin
    IntReturn := PARSE_NOT_FOUND;
  end {if}
  else
  begin
    {find the tag}
    StgTagPos := StrPos ( StgCommandString, StgUseTag );

    {skip the tag portion}
    IntSkip := StrLen ( StgUseTag ) - 1;
    StgEndTag := @StgTagPos [ IntSkip ];

    {skip the spaces}
    IntCount := 0;
    while ( StgEndTag [ IntCount ] = ' ' ) and ( StgEndTag [ IntCount ] <> #0 ) do
    begin
      IntCount := IntCount + 1;
    end; {while}

    {create the argument string}
    GetMem ( StgArgument, 81 );
    IntIndex := 0;
    while ( StgEndtag [ IntCount ] <> ' ' ) and ( StgEndtag [ IntCount ] <> #0 ) do
    begin
      StgArgument [ IntIndex ] := StgEndTag [ IntCount ];
      IntIndex := IntIndex + 1;
      IntCount := IntCount + 1;
    end; {while}

    {add the null terminator}
    StgArgument [ IntIndex ] := #0;

  end; {else}

  {garbage collection}
  FreeMem ( StgTrimmed );
  FreeMem ( StgCommandString );
  FreeMem ( StgUseTag );

  GetArgumentStg := IntReturn;

end; {GetArgumentStg}
   

{
////////////////////////////////////////////////////////////
// function GetArgumentBln
//
// returns the boolean argument following the tag StgTag
// into StgArgument. Regards "true", and "on" as true, and
// "false" and "off" as false.
//
// returns PARSE_NOT_FOUND if StgTag is not present in 
//  StgSource
// returns PARSE_UNKNOWN_ARGUMENT if the argument is not
// "true", "on", "false" or "off"
//
////////////////////////////////////////////////////////////
}

function GetArgumentBln ( StgSource : PChar; StgTag : PChar; var BlnArgument : boolean ) : integer;

var

  IntReturn : integer;
  StgArgument : PChar;

begin

  {get the string argument first}
  IntReturn := GetArgumentStg ( StgSource, StgTag, StgArgument );

  if IntReturn = SUCCESS then
  begin

    {convert it to a bool}
    IntReturn := TokenToBool ( StgArgument, BlnArgument );

    {garbage collection}
    FreeMem ( StgArgument );
  end; {if}

  GetArgumentBln := IntReturn;

end; {GetArgumentBln}


{
////////////////////////////////////////////////////////////
// function GetArgumentInt
//
// returns the long argument following the tag StgTag
// into StgArgument.
//
// returns PARSE_NOT_FOUND if StgTag is not present in 
//  StgSource
//
////////////////////////////////////////////////////////////
}

function GetArgumentInt ( StgSource : PChar; StgTag : PChar; var IntArgument : integer ) : integer;

var

  IntReturn : integer;
  StgArgument : PChar;
  StgArg : AnsiString;

begin

  IntReturn := GetArgumentStg ( StgSource, StgTag, StgArgument );
  if IntReturn = SUCCESS then
  begin
    {convert the string to an int}
    {convert the C string to a P string first}
    StgArg := StgArgument;

    IntArgument := StrToInt ( StgArg );

    {garbage collection}
    FreeMem ( StgArgument );
  end; {if}

  GetArgumentInt := IntReturn;

end; {GetArgumentInt}


{
////////////////////////////////////////////////////////////
// function GetArgumentFlt
//
// returns the double argument following the tag StgTag
// into StgArgument.
//
// returns PARSE_NOT_FOUND if StgTag is not present in 
//  StgSource
//
////////////////////////////////////////////////////////////
}

function GetArgumentFlt ( StgSource : PChar; StgTag : PChar; var FltArgument : real ) : integer;

var

  IntReturn : integer;
  StgArgument : PChar;
  StgArg : AnsiString;

begin

  IntReturn := GetArgumentStg ( StgSource, StgTag, StgArgument );
  if IntReturn = SUCCESS then
  begin
    {convert the string to a float}
    {convert the C string to a P string first}
    StgArg := StgArgument;

    FltArgument := StrToFloat ( StgArg );

    {Garbage collection}
    FreeMem ( StgArgument );
  end; {if}

  GetArgumentFlt := IntReturn;

end; {GetArgumentFlt}


{
////////////////////////////////////////////////////////////
// function ParseArray
//
// parses a delimited array into an array of strings. Array
// must be started by LIST_START, ended by LIST_END, and
// delimited by ITEM_DELIM. Assumes that the first char of
// the string is the LIST_START character
//
////////////////////////////////////////////////////////////
}

function ParseArray ( StgArray : PChar; var IntArgumentCount : integer ) : variant;

var

  VarArguments : Variant;
  IntDelims : integer;
  IntCurrChar : integer;
  IntNumStrings : integer;
  IntCurrToken: integer;
  IntStringChar : integer;
  StgTemp : PChar;

begin

  {count the number of delimiters}
  IntDelims := 0;
  IntCurrChar := 0;
  while StgArray [ IntCurrChar ] <> LIST_END do
  begin
    if StgArray [ IntCurrChar ] = ITEM_DELIM then
      IntDelims := IntDelims + 1;
    IntCurrChar := IntCurrChar + 1;
  end; {while}

  {create the array of strings}
  IntNumStrings := IntDelims + 1;
  IntArgumentCount := IntNumStrings;
  VarArguments := VarArrayCreate ( [1, IntNumStrings], varOleStr );
  GetMem ( StgTemp, 255);

  {now parse the array}
  IntCurrChar := 1;
  IntCurrToken := 1;
  while StgArray [ IntCurrChar ] <> LIST_END do
  begin
    IntStringChar := 0;
    while ( StgArray [ IntCurrChar ] <> ITEM_DELIM ) and ( StgArray [ IntCurrChar ] <> LIST_END ) do
    begin
      StgTemp [ IntStringChar ] := StgArray [ IntCurrChar ];
      IntStringChar := IntStringChar + 1;
      IntCurrChar := IntCurrChar + 1;
    end; {while}
    if StgArray [ IntCurrChar ] <> LIST_END then
      IntCurrChar := IntCurrChar + 1;

    {append the null charactor to the curent string}
    StgTemp [ IntStringChar ] := #0;

    {assign the string}
    VarArguments [ IntCurrToken ] := StrPas (StgTemp);
    IntCurrToken := IntCurrToken + 1;
  end; {while}

  {garbage collection}
  FreeMem ( StgTemp );

  ParseArray := VarArguments;
 
end; {ParseArray}


{
////////////////////////////////////////////////////////////
// function GetStgArguments
//
// returns an array of strings following StgTag in StgSource
// LngArgumentCount is the number of arguments returned.
//
// returns PARSE_NOT_FOUND if StgTag is not present in 
//  StgSource
//
////////////////////////////////////////////////////////////
}

function GetStgArguments ( StgSource : PChar; StgTag : PChar; var DyaStgArguments : Variant; var IntArgumentCount : integer ) : integer;

var

  StgTrimmed : PChar;
  StgUsetag : PChar;
  IntReturn : integer;
  StgCommandString : PChar;
  StgTagPos : PChar;
  IntSkip : integer;
  StgEndTag : PChar;
  IntCurrChar : integer;
 
begin

  StgTrimmed := AllTrim ( StgTag );
  StgUseTag := Pad ( StgTrimmed );
  IntReturn := SUCCESS;
  StgCommandString := Pad ( StgSource );
  
  if not TagPresent ( StgCommandString, StgUseTag ) then
    IntReturn := PARSE_NOT_FOUND
  else
  begin
    StgTagPos := StrPos ( StgCommandString, StgUsetag );
    IntSkip := StrLen ( StguseTag ) - 1;
    StgEndtag := @StgTagPos [ IntSkip ];

    {find the start of the array}
    IntCurrChar := 0;
    while StgEndtag [ IntCurrChar ] <> LIST_START do
      IntCurrChar := IntCurrChar + 1;

    DyaStgArguments := ParseArray ( @StgEndTag [ IntCurrChar ], IntArgumentCount );
  end; {else}
  
  {garbage collection}
  FreeMem ( StgTrimmed );
  FreeMem ( StgUseTag );
  FreeMem ( StgCommandString );

  GetStgArguments := IntReturn;

end; {GetStgArguments}


{
////////////////////////////////////////////////////////////
// function GetBlnArguments
//
// returns an array of bools following StgTag in StgSource
// LngArgumentCount is the number of arguments returned.
//
// returns PARSE_NOT_FOUND if StgTag is not present in 
//  StgSource
//
////////////////////////////////////////////////////////////
}

function GetBlnArguments ( StgSource : PChar; StgTag : PChar; var DyaBlnArguments : Variant; var IntArgumentCount : integer ) : integer;

var

  DyaStgArguments : Variant;
  IntReturn : integer;
  IntToken : integer;
  StgTemp : AnsiString;
  StgToken : array [0..4096] of char;
  BlnTemp : boolean;

begin

  IntReturn := GetStgArguments ( StgSource, StgTag, DyaStgArguments, IntArgumentCount );

  if IntReturn = SUCCESS then
  begin
    {create an array of bools}
    DyaBlnArguments := VarArrayCreate ( [1, IntArgumentCount], varBoolean );

    {populate it with the results of the previous call}
    IntToken := 1;
    while (IntToken <= IntArgumentCount) and (IntReturn = SUCCESS ) do
    begin
      StgTemp := DyaStgArguments [ IntToken ];
      StrPCopyWorks ( StgToken, StgTemp );
      IntReturn := TokenToBool ( StgToken, BlnTemp );
      DyaBlnArguments [ IntToken ] := BlnTemp;
      IntToken := IntToken + 1;
    end; {while}
  end; {if}
  GetBlnArguments := IntReturn;

end; {GetBlnArguments}


{
////////////////////////////////////////////////////////////
// function GetIntArguments
//
// returns an array of longs following StgTag in StgSource
// LngArgumentCount is the number of arguments returned.
//
// returns PARSE_NOT_FOUND if StgTag is not present in 
//  StgSource
//
////////////////////////////////////////////////////////////
}

function GetIntArguments ( StgSource : PChar; StgTag : PChar; var DyaIntArguments : Variant; var IntArgumentCount : integer ) : integer;

var

  DyaStgArguments : Variant;
  IntReturn : integer;
  IntToken : integer;

begin

  IntReturn := GetStgArguments ( StgSource, StgTag, DyaStgArguments, IntArgumentCount );
  if IntReturn = SUCCESS then
  begin
    {create an array of integers}
    DyaIntArguments := VarArrayCreate ( [1, IntArgumentCount], varInteger );

    {populate it with the results of the previous call}
    for IntToken := 1 to IntArgumentCount do
    begin
      DyaIntArguments [ IntToken ] := StrToFloat ( DyaStgArguments [ IntToken ] );
    end; {for}
  end; {if}

  GetIntArguments := IntReturn;

end; {GetIntArguments}  


{
////////////////////////////////////////////////////////////
// function GetFltArguments
//
// returns an array of doubles following StgTag in StgSource
// LngArgumentCount is the number of arguments returned.
//
// returns PARSE_NOT_FOUND if StgTag is not present in 
//  StgSource
//
////////////////////////////////////////////////////////////
}

function GetFltArguments ( StgSource : PChar; StgTag : PChar; var DyaFltArguments : Variant; var IntArgumentCount : integer ) : integer;

var

  DyaStgArguments : Variant;
  IntReturn : integer;
  IntToken : integer;

begin

  IntReturn := GetStgArguments ( StgSource, StgTag, DyaStgArguments, IntArgumentCount );
  if IntReturn = SUCCESS then
  begin
    {create an array of floats}
    DyaFltArguments := VarArrayCreate ( [1, IntArgumentCount], varDouble );

    {populate it with the results of the previous call}
    for IntToken := 1 to IntArgumentCount do
    begin
      DyaFltArguments [ IntToken ] := StrToFloat ( DyaStgArguments [ IntToken ] );
    end; {for}
  end; {if}

  GetFltArguments := IntReturn;

end; {GetFltArguments}  


end.
