unit Iris;

interface

uses
  {need to add FC3Lib and ParseFunction to the uses section of the form}
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, FC3Lib, ParseFunction,
  StdCtrls;

type
  TFrmIrisClassMain = class(TForm)
    EdtSepalLength: TEdit;
    EdtSepalWidth: TEdit;
    EdtPetalLength: TEdit;
    EdtPetalWidth: TEdit;
    LblSepalLengthCaption: TLabel;
    LblSepalWidthCaption: TLabel;
    LblPetalLegnthCaption: TLabel;
    LblPetalWidthCaption: TLabel;
    BtnClassify: TButton;
    LblClassificationCaption: TLabel;
    LblClassification: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure BtnClassifyClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  FrmIrisClassMain: TFrmIrisClassMain;

{declare some constants for the aliases of the FuzzyCOPE objects we'll}
{be using}
const
  TheMLP : string = 'TheMLP';  {alias of the MLP}
  RawData : string = 'RawData'; {alias of object to put user data into}
  InputData : string = 'InputData'; {alias of object to put normalised data into}
  OutputData : string = 'OutputData'; {alias of output data object}
  
implementation

{$R *.DFM}


procedure TFrmIrisClassMain.FormCreate(Sender: TObject);
{on creating the form, load the MLP into memory}
{and set up some other object's we'll need}

var
  IntReturn : integer; {error return code}
  StgError : string; {error message string}

begin

  {load the MLP network}
  IntReturn := LoadMLP ( TheMLP, 'iris-MLP.wgt' );

  {if the function is not successful, display the error message}
  {and exit the procedure}
  if IntReturn < 0 then
  begin
    StgError := fcGetErrorString;
    ShowMessage ( StgError );
    Exit;
  end; {if}

  {create one of the data objects we'll need:}
  {this one is for the data the user types in}
  {Don't need to create an object for the InputData to the MLP}
  {or the output data the MLP generates, as the normalisation}
  {and recall functions will do that automatically}

  {raw data needs no outputs or labels, and only needs one row}
  {because we're only typing in one example at a time}
  IntReturn := CreateData ( RawData, 4, 0, 1, false );
  if IntReturn < 0 then
  begin
    StgError := fcGetErrorString;
    ShowMessage ( StgError );
    Exit;
  end; {if}

end; {form create}


procedure TFrmIrisClassMain.FormClose(Sender: TObject; var Action: TCloseAction);

{when we close the form, we need to remove any objects still in memory}
var
  IntReturn :integer; {error return code}
  StgError : string; {error message string}

begin
  {before deleting an object, check that it exists}

  {first the MLP}
  if AliasExists ( TheMLP ) then
  begin
    IntReturn := DeleteObject ( TheMLP );
    if IntReturn < 0 then {an error occurred}
    begin
      StgError := fcGetErrorString;
      ShowMessage ( StgError );
    end; {if IntReturn < 0}
  end; {if exists}

  {then the raw data object}
  if AliasExists ( RawData ) then
  begin
    IntReturn := DeleteObject ( RawData );
    if IntReturn < 0 then {an error occurred}
    begin
      StgError := fcGetErrorString;
      ShowMessage ( StgError );
    end; {if IntReturn < 0}
  end; {if exists}

  {then the normalised data object}
  if AliasExists ( InputData ) then
  begin
    IntReturn := DeleteObject ( InputData );
    if IntReturn < 0 then {an error occurred}
    begin
      StgError := fcGetErrorString;
      ShowMessage ( StgError );
    end; {if IntReturn < 0}
  end; {if exists}

  {finally the output data object}
  if AliasExists ( OutputData ) then
  begin
    IntReturn := DeleteObject ( OutputData );
    if IntReturn < 0 then {an error occurred}
    begin
      StgError := fcGetErrorString;
      ShowMessage ( StgError );
    end; {if IntReturn < 0}
  end; {if exists}

end; {form unload}


procedure TFrmIrisClassMain.BtnClassifyClick(Sender: TObject);

var
  IntReturn : integer; {error return code}
  StgError : string; {error message string}
  DyaFltRawInputs : Variant; {input data row}
  DyaFltOutputs : Variant; {outputs data row}
  DyaStgLabels : Variant; {row of labels}
  DyaFltLabelWeights : Variant; {row of label weightings}
  DyaFltInputMax : Variant; {input maximums}
  DyaFltInputMin : Variant; {inputs minimums}
  DyaFltInputs : Variant; {input row returned from output}
  IntInputs : integer; {number of inputs returned by GetNextRow}
  IntOutputs : integer; {number of outputs returned by GetNextRow}
  IntNumLabels : integer; {number of lables returned by GetNextRow}

begin

  {first we need to check that the text boxes all have data in them}
  {if any of the boxes are empty, display an error message and quit}
  {the procedure}
  if EdtSepalLength.text = '' then
  begin
    ShowMessage ( 'Please enter a number in the Sepal Length box' );
    Exit;
  end; {if}

  if EdtSepalWidth.text = '' then
  begin
    ShowMessage ( 'Please enter a number in the Sepal Width box' );
    Exit;
  end; {if}

  if EdtPetalLength.text = '' then
  begin
    ShowMessage ( 'Please enter a number in the Petal Length box' );
    Exit;
  end; {if}

  if EdtPetalWidth.text = '' then
  begin
    ShowMessage ( 'Please enter a number in the Petal Width box' );
    Exit;
  end; {if}

  {having checked that the data entry boxes aren't empty}
  {we need to create the input data row}
  DyaFltRawInputs := VarArrayCreate ( [1, 4], varDouble );

  {then retrieve data from the interface and insert it into the row}
  {need to convert the text to floats to go into the array / row }
  DyaFltRawInputs [ 1 ] := StrToFloat ( EdtSepalLength.text );
  DyaFltRawInputs [ 2 ] := StrToFloat ( EdtSepalWidth.text );
  DyaFltRawInputs [ 3 ] := StrToFloat ( EdtPetalLength.text );
  DyaFltRawInputs [ 4 ] := StrToFloat ( EdtPetalWidth.text );

  {append the row into the raw input data object}
  {don't need to initialise the label and outputs}
  {arrays because we know that the data object doesn't}
  {have outputs or labelling enabled, and we can tell}
  {the function to ignore the output and label rows anyway}
  IntReturn := AppendRow ( RawData, DyaFltRawInputs, DyaFltOutputs, DyaStgLabels, DyaFltLabelWeights, true, false, false );
  if IntReturn < 0 then {an error occurred}
  begin
    StgError := fcGetErrorString;
    ShowMessage ( StgError );
    Exit;
  end; {if}

  {normalise the data into the InputData object}
  {delete the InputData object first if it exists}
  {otherwise the  normalisation function will fail}
  if AliasExists ( InputData ) then
  begin
    IntReturn := DeleteObject ( InputData );
    if IntReturn < 0 then {an error occurred}
    begin
      StgError := fcGetErrorString;
      ShowMessage ( StgError );
      Exit;
    end; {if IntReturn < 0}
  end; {if exists}

  {now normalise the raw data}
  {set up the input maximums and minimums first}
  {Note that if the entered data exceeds the maximum}
  {or minimum then the normalised value will simply be}
  {truncated to the range [0, 1]}
  DyaFltInputMax := VarArrayCreate ( [1, 4], varDouble );
  DyaFltInputMin := VarArrayCreate ( [1, 4], varDouble );
  DyaFltInputMax [ 1 ] := 7.9;
  DyaFltInputMax [ 2 ] := 4.4;
  DyaFltInputMax [ 3 ] := 6.9;
  DyaFltInputMax [ 4 ] := 2.5;
  DyaFltInputMin [ 1 ] := 4.3;
  DyaFltInputMin [ 2 ] := 2.0;
  DyaFltInputMin [ 3 ] := 1.0;
  DyaFltInputMin [ 4 ] := 1.0;

  {call the normalisation function for the inputs: no outputs to normalise}
  IntReturn := LinearNormaliseInputs ( RawData, InputData, DyaFltInputMin, DyaFltInputMax );
  if IntReturn < 0 then {an error occurred}
  begin
    StgError := fcGetErrorString;
    ShowMessage ( StgError );
    Exit;
  end; {if}

  {prepare to recall with the normalised data}
  {if the output data exists, delete it}
  if AliasExists ( OutputData ) then
  begin
    IntReturn := DeleteObject ( OutputData );
    if IntReturn < 0 then {an error occurred}
    begin
      StgError := fcGetErrorString;
      ShowMessage ( StgError );
      Exit;
    end; {if IntReturn < 0}
  end; {if exists}

  {perform the recall}
  IntReturn := Recall ( TheMLP, InputData, OutputData );
  if IntReturn < 0 then {an error occurred}
  begin
    StgError := fcGetErrorString;
    ShowMessage ( StgError );
    Exit;
  end; {if}

  {retrieve the output row of the resulting data object}
  {first reset the data record pointer}
  IntReturn := ResetData ( OutputData );
  if IntReturn < 0 then {an error occurred}
  begin
    StgError := fcGetErrorString;
    ShowMessage ( StgError );
    Exit;
  end; {if}

  {then retrieve the row. We can re-use most of the rows we defined before}
  {as an output data object only has an output table}
  IntReturn := GetNextRow ( OutputData, DyaFltInputs, DyaFltOutputs, DyaStgLabels, DyaFltLabelWeights, IntInputs, IntOutputs, IntNumLabels );
  if IntReturn < 0 then {an error occurred}
  begin
    StgError := fcGetErrorString;
    ShowMessage ( StgError );
    Exit;
  end; {if}

  {we are only interested in the output data row, so we}
  {examine each entry in it to determine the iris' classification}
  {output 1 = Iris Setosa}
  {output 2 = Iris Versicolour}
  {output 3 = Iris Virginica}
  if ( DyaFltOutputs [ 1 ] > DyaFltOutputs [ 2 ] ) and ( DyaFltOutputs [ 1 ] > DyaFltOutputs [ 3 ] ) then
  {Iris is type Setosa}
  begin
    LblClassification.caption := 'Iris Setosa';
  end {if}
  else if ( DyaFltOutputs [ 2 ] > DyaFltOutputs [ 1 ] ) and ( DyaFltOutputs [ 2 ] > DyaFltOutputs [ 3 ] ) then
  {Iris is type Versicolour}
  begin {if}
    LblClassification.caption := 'Iris Versicolour';
  end {else if}
  else if ( DyaFltOutputs [ 3 ] > DyaFltOutputs [ 1 ] ) and ( DyaFltOutputs [ 3 ] > DyaFltOutputs [ 2 ] ) then
  {Iris is type Virginica}
  begin
    LblClassification.caption := 'Iris Virginica';
  end {else if}
  else
  {MLP can't classify the data at all}
  begin
    LblClassification.caption := 'Unknown';
  end; {else}

end; {click}

end.
